<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Workflow;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\OfferDate;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Workflow\Attribute\AsGuardListener;
use Symfony\Component\Workflow\Event\GuardEvent;

class AttendanceGuard
{
    public function __construct(private readonly Security $security, #[Autowire(param: 'ferienpass.withdraw_grace_period')] private readonly ?string $gracePeriod, #[Autowire(param: 'ferienpass.no_withdraw_after_grace_period')] private readonly bool $noWithdrawsAfterGracePeriod, #[Autowire(param: 'ferienpass.no_withdraw_after_application_deadline')] private readonly bool $noWithdrawsAfterApplicationDeadline)
    {
    }

    #[AsGuardListener('attendance', transition: Attendance::TRANSITION_WITHDRAW)]
    public function withdraw(GuardEvent $event): void
    {
        $attendance = $event->getSubject();
        if (!$attendance instanceof Attendance) {
            throw new \RuntimeException();
        }

        $now = new \DateTimeImmutable();
        $offer = $attendance->getOffer();

        if ($this->security->isGranted('ROLE_ADMIN')) {
            return;
        }

        if ($offer->isCancelled()) {
            $event->setBlocked(true);

            return;
        }

        $offerDate = $offer->getDates()->first();
        if ($offerDate instanceof OfferDate && $now > $offerDate->getBegin()) {
            $event->setBlocked(true);

            return;
        }

        $now = new \DateTimeImmutable();
        if ($this->noWithdrawsAfterApplicationDeadline && $offer->getApplicationDeadline() && $now > $offer->getApplicationDeadline()) {
            $event->setBlocked(true);

            return;
        }

        if ($this->noWithdrawsAfterGracePeriod && !$this->inGracePeriod($attendance->getOffer())) {
            $event->setBlocked(true);
        }
    }

    private function inGracePeriod(OfferInterface $offer): bool
    {
        $gracePeriod = $this->gracePeriod ? new \DateTimeImmutable($this->gracePeriod) : null;

        return !$gracePeriod instanceof \DateTimeImmutable || ($offer->getDates()->isEmpty() || $offer->getDates()->first()->getBegin() > $gracePeriod);
    }
}
