<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Security\Voter;

use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Host;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\Voter\Voter;

class OfferVoter extends Voter
{
    public function __construct(private readonly Security $security, private readonly HostRepository $hosts, #[Autowire(param: 'ferienpass_admin.participant_list.hosts_can_add')] private readonly bool $hostsCanAddParticipants = false, #[Autowire(param: 'ferienpass_admin.participant_list.hosts_can_edit')] private readonly bool $hostsCanEditParticipantList = false)
    {
    }

    protected function supports($attribute, $subject): bool
    {
        $operations = [
            'view',
            'create',
            'edit',
            'copy',
            'delete',
            OfferInterface::TRANSITION_FINALIZE,
            OfferInterface::TRANSITION_TO_DRAFT,
            OfferInterface::TRANSITION_APPROVE,
            OfferInterface::TRANSITION_PUBLISH,
            OfferInterface::TRANSITION_UNPUBLISH,
            OfferInterface::TRANSITION_RELAUNCH,
            OfferInterface::TRANSITION_CANCEL,
            OfferInterface::TRANSITION_COMPLETE,
            'participants.view',
            'participants.add',
            'participants.edit',
            'participants.reject',
            'participants.confirm',
        ];

        return $subject instanceof OfferInterface && \in_array($attribute, $operations, true);
    }

    protected function voteOnAttribute($attribute, $subject, TokenInterface $token, ?\Symfony\Component\Security\Core\Authorization\Voter\Vote $vote = null): bool
    {
        $user = $token->getUser();
        if (!$user instanceof User) {
            return false;
        }

        /** @var OfferInterface $offer */
        $offer = $subject;

        return match ($attribute) {
            'view' => $this->canView($offer, $user),
            'edit' => $this->canEdit($offer, $user),
            'create' => $this->canCreate($offer, $user),
            'copy' => $this->canCopy($offer, $user),
            'delete' => $this->canDelete($offer, $user),
            OfferInterface::TRANSITION_CANCEL => $this->canCancel($offer, $user),
            OfferInterface::TRANSITION_RELAUNCH => $this->canRelaunch($offer, $user),
            OfferInterface::TRANSITION_PUBLISH => $this->canPublish(),
            OfferInterface::TRANSITION_UNPUBLISH => $this->canUnPublish(),
            OfferInterface::TRANSITION_APPROVE, OfferInterface::TRANSITION_UNAPPROVE => $this->canApprove(),
            OfferInterface::TRANSITION_FINALIZE => $this->canFinalize($offer, $user),
            OfferInterface::TRANSITION_TO_DRAFT => $this->canSetToDraft(),
            OfferInterface::TRANSITION_COMPLETE => $this->canComplete($offer, $user),
            'participants.view' => $this->canViewParticipants($offer, $user),
            'participants.add' => $this->canAddParticipants($offer, $user),
            'participants.edit', 'participants.reject', 'participants.confirm' => $this->canEditParticipants($offer, $user),
            default => throw new \LogicException('This code should not be reached!'),
        };
    }

    private function canView(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        $userHosts = $this->hosts->findByUser($user);
        $userHostIds = array_map(fn (Host $host) => $host->getId(), $userHosts);

        return $offer->getHosts()->filter(fn (Host $host) => \in_array($host->getId(), $userHostIds, true))->count() > 0;
    }

    private function canEdit(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        if (false === $this->canView($offer, $user)) {
            return false;
        }

        if ($offer->isPublished() || $offer->isReviewed()) {
            return false;
        }

        $edition = $offer->getEdition();

        return $edition->isEditableForHosts() && $this->userIsAllowedHost($user, $edition);
    }

    private function canCreate(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        $edition = $offer->getEdition();

        // Non-admins are not allowed to create on root leve
        if (!$edition) {
            return false;
        }

        if (false === $this->userIsAllowedHost($user, $offer->getEdition())) {
            return false;
        }

        return !$edition->getActiveTasks('host_editing_stage')->isEmpty();
    }

    private function userIsAllowedHost(User $user, Edition $edition): bool
    {
        if ($edition->getHosts()->isEmpty()) {
            return true;
        }

        $userHosts = $this->hosts->findByUser($user);
        $userHostIds = array_map(fn (Host $host) => $host->getId(), $userHosts);

        return $edition->getHosts()->filter(fn (Host $host) => \in_array($host->getId(), $userHostIds, true))->count() > 0;
    }

    private function canCopy(OfferInterface $offer, User $user): bool
    {
        return $this->canCreate($offer, $user) && $this->canView($offer, $user);
    }

    private function canDelete(OfferInterface $offer, User $user): bool
    {
        if (false === $this->canView($offer, $user)) {
            return false;
        }

        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        if (!$offer->getAttendances()->isEmpty()) {
            return false;
        }

        $edition = $offer->getEdition();

        return $edition->isEditableForHosts() && $this->userIsAllowedHost($user, $edition);
    }

    private function canCancel(OfferInterface $offer, User $user): bool
    {
        if (false === $this->canView($offer, $user)) {
            return false;
        }

        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        return $this->hostsCanEditParticipantList;
    }

    private function canRelaunch(OfferInterface $offer, User $user): bool
    {
        return $this->canCancel($offer, $user);
    }

    private function canApprove(): bool
    {
        return $this->security->isGranted('ROLE_ADMIN');
    }

    private function canPublish(): bool
    {
        return $this->security->isGranted('ROLE_ADMIN');
    }

    private function canUnPublish(): bool
    {
        return $this->security->isGranted('ROLE_ADMIN');
    }

    private function canFinalize(OfferInterface $offer, User $user): bool
    {
        return $this->canView($offer, $user);
    }

    private function canSetToDraft(): bool
    {
        return $this->security->isGranted('ROLE_ADMIN');
    }

    private function canComplete(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        return $this->canView($offer, $user);
    }

    private function canViewParticipants(OfferInterface $offer, User $user): bool
    {
        $edition = $offer->getEdition();
        if (null === $edition) {
            return false;
        }

        if (!$edition->isParticipantListReleased() && !$this->security->isGranted('ROLE_ADMIN')) {
            return false;
        }

        return $this->canView($offer, $user);
    }

    private function canAddParticipants(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        if (false === $this->canView($offer, $user)) {
            return false;
        }

        return $this->hostsCanAddParticipants;
    }

    private function canEditParticipants(OfferInterface $offer, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        if (false === $this->canView($offer, $user)) {
            return false;
        }

        return $this->hostsCanEditParticipantList;
    }
}
