<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Security\Voter;

use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Host;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\Voter\Voter;

class EditionVoter extends Voter
{
    public function __construct(private readonly Security $security, private readonly HostRepository $hosts)
    {
    }

    protected function supports($attribute, $subject): bool
    {
        if (!\in_array($attribute, ['view', 'edit', 'stats', 'offer.create', 'delete', 'assign', 'lotting'], true)) {
            return false;
        }

        return $subject instanceof Edition;
    }

    protected function voteOnAttribute($attribute, $subject, TokenInterface $token, ?\Symfony\Component\Security\Core\Authorization\Voter\Vote $vote = null): bool
    {
        $user = $token->getUser();
        if (!$user instanceof User) {
            return false;
        }

        /** @var Edition $edition */
        $edition = $subject;

        return match ($attribute) {
            'view', 'stats', 'assign' => $this->canView(),
            'edit' => $this->canEdit(),
            'delete' => $this->canDelete(),
            'offer.create' => $this->canCreateOffer($edition, $user),
            'lotting' => $this->canLotting(),
            default => throw new \LogicException('This code should not be reached!'),
        };
    }

    private function canView(): bool
    {
        return $this->security->isGranted('ROLE_ADMIN');
    }

    private function canEdit(): bool
    {
        return $this->canView();
    }

    private function canDelete(): bool
    {
        return $this->canEdit();
    }

    private function canLotting(): bool
    {
        return $this->canEdit();
    }

    private function canCreateOffer(Edition $edition, User $user): bool
    {
        if ($this->security->isGranted('ROLE_ADMIN')) {
            return true;
        }

        if (false === $this->userIsAllowedHost($user, $edition)) {
            return false;
        }

        return $edition->isEditableForHosts();
    }

    private function userIsAllowedHost(User $user, Edition $edition): bool
    {
        if ($edition->getHosts()->isEmpty()) {
            return true;
        }

        $userHosts = $this->hosts->findByUser($user);
        $userHostIds = array_map(fn (Host $host) => $host->getId(), $userHosts);

        return $edition->getHosts()->filter(fn (Host $host) => \in_array($host->getId(), $userHostIds, true))->count() > 0;
    }
}
