<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Security;

use Ferienpass\CoreBundle\Entity\ApiToken;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\ApiTokenRepository;
use Symfony\Component\PasswordHasher\Hasher\PasswordHasherFactoryInterface;
use Symfony\Component\Security\Core\Exception\BadCredentialsException;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Http\AccessToken\AccessTokenHandlerInterface;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;

class ApiTokenHandler implements AccessTokenHandlerInterface
{
    public function __construct(private readonly ApiTokenRepository $apiTokenRepository, private readonly PasswordHasherFactoryInterface $hasherFactory)
    {
    }

    public function getUserBadgeFrom(#[\SensitiveParameter] string $accessToken): UserBadge
    {
        if (!str_starts_with($accessToken, ApiToken::PERSONAL_ACCESS_TOKEN_PREFIX)) {
            throw new BadCredentialsException();
        }

        if (1 !== substr_count($accessToken, '.')) {
            throw new BadCredentialsException();
        }

        [$locator, $secret] = explode('.', $accessToken, 2);

        /** @var ApiToken $token */
        $token = $this->apiTokenRepository->findOneBy(['locator' => $locator]);
        if (!$token || !$this->isTokenValid($token->getOwnedBy(), $token, $secret)) {
            throw new BadCredentialsException();
        }

        if (!$token->isValid()) {
            throw new CustomUserMessageAuthenticationException('Token expired');
        }

        $token->getOwnedBy()->markAsTokenAuthenticated($token->getScopes());

        return new UserBadge($token->getOwnedBy()->getUserIdentifier());
    }

    private function isTokenValid(User $user, ApiToken $token, #[\SensitiveParameter] $secret): bool
    {
        return $this->hasherFactory->getPasswordHasher($user)->verify($token->getSecret(), $secret);
    }
}
