<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Repository;

use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\Persistence\ManagerRegistry;
use Ferienpass\CoreBundle\Entity\SentMessage;
use Ferienpass\CoreBundle\Entity\SentMessageEngagement;
use Ferienpass\CoreBundle\Entity\User;
use Symfony\Component\RemoteEvent\Event\Mailer\MailerDeliveryEvent;
use Symfony\Component\RemoteEvent\Event\Mailer\MailerEngagementEvent;

class SentMessageRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, SentMessage::class);
    }

    public function findByUser(User $user): array
    {
        $qb = $this->createQueryBuilder('message')
            ->innerJoin('message.logEntry', 'log')
            ->innerJoin('log.users', 'user', Join::WITH, 'user = :user')
            ->addOrderBy('message.createdAt', 'DESC')
            ->setParameter('user', $user)
            ->getQuery()
        ;

        return $qb->getResult();
    }

    public function findWithError(?string $since = null)
    {
        return $this->findByEngagement([
            MailerDeliveryEvent::BOUNCE,
            MailerDeliveryEvent::DROPPED,
            MailerDeliveryEvent::DEFERRED,
            MailerEngagementEvent::UNSUBSCRIBE,
            MailerEngagementEvent::SPAM,
        ], $since);
    }

    public function findByEngagement(array $events, ?string $since = null): array
    {
        $qb = $this->createQueryBuilder('message')
            ->where('EXISTS (
        SELECT 1 FROM '.SentMessageEngagement::class.' e
        WHERE e.message = message AND e.event IN (:events)
    )')
            ->setParameter('events', $events)
            ->addOrderBy('message.createdAt', 'DESC')
        ;

        if (null !== $since) {
            $qb
                ->andWhere('message.createdAt >= :since')
                ->setParameter('since', new \DateTimeImmutable('now')->modify("-$since"), Types::DATETIME_IMMUTABLE);
        }

        return $qb->getQuery()->getResult();
    }
}
