<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Repository;

use DeepCopy\DeepCopy;
use DeepCopy\Filter\ChainableFilter;
use DeepCopy\Filter\Doctrine\DoctrineCollectionFilter;
use DeepCopy\Filter\Doctrine\DoctrineEmptyCollectionFilter;
use DeepCopy\Filter\Doctrine\DoctrineProxyFilter;
use DeepCopy\Filter\KeepFilter;
use DeepCopy\Filter\ReplaceFilter;
use DeepCopy\Filter\SetNullFilter;
use DeepCopy\Matcher\Doctrine\DoctrineProxyMatcher;
use DeepCopy\Matcher\PropertyMatcher;
use DeepCopy\Matcher\PropertyTypeMatcher;
use Doctrine\Common\Collections\Collection;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\OfferAttachment;
use Ferienpass\CoreBundle\Entity\OfferMedia;
use Symfony\Component\Uid\Uuid;

class OfferRepository extends EntityRepository implements OfferRepositoryInterface
{
    public function findByAlias(string $alias): ?OfferInterface
    {
        return $this->findOneBy(['alias' => $alias]);
    }

    public function findByUuid(string|Uuid $uuid): ?OfferInterface
    {
        return $this->findOneBy(['uuid' => $uuid]);
    }

    public function createCopy(OfferInterface $original): OfferInterface
    {
        $copier = new DeepCopy();

        $copier->addFilter(new ChainableFilter(new DoctrineProxyFilter()), new DoctrineProxyMatcher());
        $copier->addFilter(new SetNullFilter(), new PropertyMatcher(OfferInterface::class, 'id'));
        $copier->addFilter(new ReplaceFilter(fn () => Uuid::v4()), new PropertyMatcher(OfferInterface::class, 'uuid'));
        $copier->addFilter(new SetNullFilter(), new PropertyMatcher(OfferInterface::class, 'alias'));
        $copier->addFilter(new SetNullFilter(), new PropertyMatcher(OfferInterface::class, 'contactUser'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferInterface::class, 'edition'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferInterface::class, 'hosts'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferInterface::class, 'variantBase'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferInterface::class, 'memberAssociations'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferInterface::class, 'categories'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferAttachment::class, 'attachment'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferAttachment::class, 'offer'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferMedia::class, 'media'));
        $copier->addFilter(new KeepFilter(), new PropertyMatcher(OfferMedia::class, 'offer'));
        $copier->addFilter(new DoctrineEmptyCollectionFilter(), new PropertyMatcher(OfferInterface::class, 'attendances'));
        $copier->addFilter(new DoctrineEmptyCollectionFilter(), new PropertyMatcher(OfferInterface::class, 'variants'));
        $copier->addFilter(new DoctrineEmptyCollectionFilter(), new PropertyMatcher(OfferInterface::class, 'activity'));
        $copier->addFilter(new DoctrineEmptyCollectionFilter(), new PropertyMatcher(OfferInterface::class, 'messengerLogs'));
        $copier->addFilter(new DoctrineEmptyCollectionFilter(), new PropertyMatcher(OfferInterface::class, 'friendCodes'));
        $copier->addFilter(new DoctrineCollectionFilter(), new PropertyTypeMatcher(Collection::class));

        /** @var OfferInterface $new */
        $new = $copier->copy($original);

        $new->resetStatus();

        $new->setVariantBase(null);

        foreach ($new->getMedia() as $media) {
            /** @var OfferMedia $media */
            $media->setOffer($new);
        }

        foreach ($new->getAttachments() as $attachment) {
            /** @var OfferAttachment $attachment */
            $attachment->setOffer($new);
        }

        return $new;
    }

    public function createVariant(OfferInterface $original): OfferInterface
    {
        $new = $this->createCopy($original);

        $new->setVariantBase($original->getVariantBase() ?? $original);

        return $new;
    }

    public function variantFields(): array
    {
        return [
            'name',
            'description',
            'meetingPoint',
            'bring',
            'minParticipants',
            'maxParticipants',
            'minAge',
            'maxAge',
            'requiresApplication',
            'onlineApplication',
            'applyText',
            'contactUser',
            'fee',
            'image',
            'edition',
            'hosts',
            'categories',
            'wheelchairAccessible',
            'requiresAgreementLetter',
            'applicationExtra',
            'media',

            // Not real fields
            'contact',
            'uploadMedia',
        ];
    }

    public function findUpcoming(): array
    {
        $qb = $this->createQueryBuilder('offer')
            ->innerJoin('offer.dates', 'date')
            ->where('date.begin > CURRENT_TIMESTAMP()')
            ->addSelect('MIN(date.begin) AS HIDDEN minBegin')
            ->groupBy('offer.id')
            ->orderBy('minBegin')
        ;

        return $qb->getQuery()->getResult();
    }
}
