<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Notifier;

use Contao\CoreBundle\Filesystem\FilesystemItem;
use Contao\CoreBundle\Filesystem\VirtualFilesystemInterface;
use Ferienpass\CoreBundle\Entity;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\DbafsAttachment;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Host;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Notification\AbstractNotification;
use Ferienpass\CoreBundle\Notification\AccountActivatedNotification;
use Ferienpass\CoreBundle\Notification\AccountCreatedNotification;
use Ferienpass\CoreBundle\Notification\AccountRegistrationHelpNotification;
use Ferienpass\CoreBundle\Notification\ApplicationsClosedNotification;
use Ferienpass\CoreBundle\Notification\AttendanceConfirmedNotification;
use Ferienpass\CoreBundle\Notification\AttendanceDecisions;
use Ferienpass\CoreBundle\Notification\AttendanceNewlyConfirmedNotification;
use Ferienpass\CoreBundle\Notification\AttendanceRejectedNotification;
use Ferienpass\CoreBundle\Notification\EmailToAwareNotificationInterface;
use Ferienpass\CoreBundle\Notification\HostCreatedNotification;
use Ferienpass\CoreBundle\Notification\OfferCancelledNotification;
use Ferienpass\CoreBundle\Notification\OfferCompletedNotification;
use Ferienpass\CoreBundle\Notification\OfferCompleteNotification;
use Ferienpass\CoreBundle\Notification\OfferFinalizedNotification;
use Ferienpass\CoreBundle\Notification\OfferRelaunchedNotification;
use Ferienpass\CoreBundle\Notification\PaymentCreatedNotification;
use Ferienpass\CoreBundle\Notification\RemindAttendanceNotification;
use Ferienpass\CoreBundle\Notification\UserInvitationNotification;
use Ferienpass\CoreBundle\Notification\UserPasswordNotification;
use Ferienpass\CoreBundle\Notifier\Message\Attachment;
use Ferienpass\CoreBundle\Repository\NotificationRepository;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\TaggedIterator;
use Symfony\Component\Notifier\Notification\Notification;
use Symfony\Component\Notifier\NotifierInterface;
use Symfony\Component\Notifier\Recipient\RecipientInterface;

class Notifier implements NotifierInterface
{
    /**
     * @var array<string, AbstractNotification>
     */
    private array $notifications;

    public function __construct(#[TaggedIterator('ferienpass.notification', defaultIndexMethod: 'getName')] iterable $notifications, private readonly NotifierInterface $notifier, private readonly NotificationRepository $notificationRepository, #[Autowire(param: 'ferienpass.reply_address')] private readonly ?string $replyTo, private readonly VirtualFilesystemInterface $attachmentsStorage)
    {
        $this->notifications = $notifications instanceof \Traversable ? iterator_to_array($notifications) : $notifications;
    }

    public function accountActivated(User $user): ?AccountActivatedNotification
    {
        $notification = $this->get(AccountActivatedNotification::getName());
        if (!$notification instanceof AccountActivatedNotification) {
            return null;
        }

        return $notification->user($user);
    }

    public function accountCreated(User $user): ?AccountCreatedNotification
    {
        $notification = $this->get(AccountCreatedNotification::getName());
        if (!$notification instanceof AccountCreatedNotification) {
            return null;
        }

        return $notification->user($user);
    }

    public function accountRegistrationHelp(User $user): ?AccountRegistrationHelpNotification
    {
        $notification = $this->get(AccountRegistrationHelpNotification::getName());
        if (!$notification instanceof AccountRegistrationHelpNotification) {
            return null;
        }

        return $notification->user($user);
    }

    public function hostCreated(Host $host, User $user): ?HostCreatedNotification
    {
        $notification = $this->get(HostCreatedNotification::getName());
        if (!$notification instanceof HostCreatedNotification) {
            return null;
        }

        return $notification->host($host)->user($user);
    }

    public function userPassword(string $token, User $user): ?UserPasswordNotification
    {
        $notification = $this->get(UserPasswordNotification::getName());
        if (!$notification instanceof UserPasswordNotification) {
            return null;
        }

        return $notification->token($token)->user($user);
    }

    public function attendanceNewlyConfirmed(Attendance $attendance): ?AttendanceNewlyConfirmedNotification
    {
        $notification = $this->get(AttendanceNewlyConfirmedNotification::getName(), $attendance->getOffer()->getEdition() ?? null);
        if (!$notification instanceof AttendanceNewlyConfirmedNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function applicationsClosed(OfferInterface $offer): ?ApplicationsClosedNotification
    {
        $notification = $this->get(ApplicationsClosedNotification::getName(), $offer->getEdition());
        if (!$notification instanceof ApplicationsClosedNotification) {
            return null;
        }

        return $notification->offer($offer);
    }

    public function attendanceConfirmed(Attendance $attendance): ?AttendanceConfirmedNotification
    {
        $notification = $this->get(AttendanceConfirmedNotification::getName(), $attendance->getOffer()->getEdition());
        if (!$notification instanceof AttendanceConfirmedNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function attendanceRejected(Attendance $attendance): ?AttendanceRejectedNotification
    {
        $notification = $this->get(AttendanceRejectedNotification::getName(), $attendance->getOffer()->getEdition());
        if (!$notification instanceof AttendanceRejectedNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function attendanceDecisions($attendances, ?Edition $edition = null): ?AttendanceDecisions
    {
        $notification = $this->get(AttendanceDecisions::getName(), $edition);
        if (!$notification instanceof AttendanceDecisions) {
            return null;
        }

        foreach ($attendances as $attendance) {
            $notification->attendance($attendance);
        }

        return $notification;
    }

    public function offerCancelled(Attendance $attendance): ?OfferCancelledNotification
    {
        $notification = $this->get(OfferCancelledNotification::getName(), $attendance->getoffer()->getEdition());
        if (!$notification instanceof OfferCancelledNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function offerFinalized(OfferInterface $offer): ?OfferFinalizedNotification
    {
        $notification = $this->get(OfferFinalizedNotification::getName(), $offer->getEdition());
        if (!$notification instanceof OfferFinalizedNotification) {
            return null;
        }

        return $notification->offer($offer);
    }

    public function offerCompleted(OfferInterface $offer): ?OfferCompletedNotification
    {
        $notification = $this->get(OfferCompletedNotification::getName(), $offer->getEdition());
        if (!$notification instanceof OfferCompletedNotification) {
            return null;
        }

        return $notification->offer($offer);
    }

    public function offerComplete(OfferInterface $offer): ?OfferCompleteNotification
    {
        $notification = $this->get(OfferCompleteNotification::getName(), $offer->getEdition());
        if (!$notification instanceof OfferCompleteNotification) {
            return null;
        }

        return $notification->offer($offer);
    }

    public function offerRelaunched(Attendance $attendance): ?OfferRelaunchedNotification
    {
        $notification = $this->get(OfferRelaunchedNotification::getName(), $attendance->getOffer()->getEdition());
        if (!$notification instanceof OfferRelaunchedNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function paymentCreated(Payment $payment): ?PaymentCreatedNotification
    {
        $notification = $this->get(PaymentCreatedNotification::getName());
        if (!$notification instanceof PaymentCreatedNotification) {
            return null;
        }

        return $notification->payment($payment);
    }

    public function remindAttendance(Attendance $attendance): ?RemindAttendanceNotification
    {
        $notification = $this->get(RemindAttendanceNotification::getName(), $attendance->getOffer()->getEdition());
        if (!$notification instanceof RemindAttendanceNotification) {
            return null;
        }

        return $notification->attendance($attendance);
    }

    public function userInvitation(User $user, Host $host, string $inviteeEmail): ?UserInvitationNotification
    {
        $notification = $this->get(UserInvitationNotification::getName());
        if (!$notification instanceof UserInvitationNotification) {
            return null;
        }

        return $notification->user($user)->host($host)->inviteeEmail($inviteeEmail);
    }

    public function send(Notification $notification, RecipientInterface ...$recipients): void
    {
        $this->notifier->send($notification, ...$recipients);
    }

    public function types(): array
    {
        return array_keys($this->notifications);
    }

    public function getClass(string $key): ?string
    {
        if (!\array_key_exists($key, $this->notifications)) {
            return null;
        }

        return $this->notifications[$key]::class;
    }

    public function createMock(string $key, string $subject, string $content): ?Notification
    {
        if (!\array_key_exists($key, $this->notifications)) {
            return null;
        }

        $notification = $this->notifications[$key];

        $notification = $notification->createMock();

        $notification
            ->subject($subject)
            ->content($content);

        return $notification;
    }

    public function get(string $key, ?Edition $edition = null, bool $strict = false): ?Notification
    {
        if (!\array_key_exists($key, $this->notifications)) {
            return null;
        }

        $notification = clone $this->notifications[$key];
        $entity = $this->notificationRepository->findOneByTypeAndEdition($key, $edition, $strict);
        if (!($entity instanceof Entity\Notification)) {
            return null;
        }

        $notification
            ->subject($entity->getEmailSubject() ?? '')
            ->content($entity->getEmailText() ?? '')
            ->smsText($entity->getSmsText() ?? '')
        ;

        if ($entity->getEdition() instanceof Edition) {
            $notification->edition($entity->getEdition());
        }

        if ($notification instanceof EmailToAwareNotificationInterface) {
            $notification->emailTo($entity->getEmailTo());
        }

        if ($this->replyTo) {
            $notification->replyTo($this->replyTo);
        }

        if ($entity->getEmailReplyTo()) {
            $notification->replyTo($entity->getEmailReplyTo());
        }

        if ($entity->getEmailAttachment() && ($attachment = $this->getAttachment($entity->getEmailAttachment())) instanceof FilesystemItem) {
            $notification->attachment(new Attachment($this->attachmentsStorage, $attachment));
        }

        return $notification;
    }

    private function getAttachment(DbafsAttachment $attachment): ?FilesystemItem
    {
        return $this->attachmentsStorage->get($attachment->getUuid());
    }
}
