<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\NotificationHandler;

use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Message\AccountCreated as AccountCreatedEvent;
use Ferienpass\CoreBundle\Notification\AccountCreatedNotification;
use Ferienpass\CoreBundle\Notifier\Notifier;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Symfony\Component\HttpFoundation\UriSigner;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Notifier\Recipient\Recipient;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RequestContextAwareInterface;
use Symfony\Component\Uid\Uuid;

#[AsMessageHandler(bus: 'event.bus')]
class AccountCreated
{
    public function __construct(private readonly Notifier $notifier, private readonly UserRepository $users, private readonly UrlGeneratorInterface $urlGenerator, private readonly UriSigner $uriSigner, private readonly PhoneFormatter $phone)
    {
    }

    public function __invoke(AccountCreatedEvent $event, Uuid $uuid): void
    {
        /** @var User $user */
        $user = $this->users->find($event->getUserId());
        if (null === $user) {
            return;
        }

        $notification = $this->notifier->accountCreated($user);
        if (!$notification instanceof AccountCreatedNotification || '' === $email = (string) $user->getEmail()) {
            return;
        }

        if ($event->getHost() && $this->urlGenerator instanceof RequestContextAwareInterface) {
            $context = $this->urlGenerator->getContext();
            $context->setHost($event->getHost());
        }

        $this->notifier->send(
            $notification->belongsTo($uuid)->actionUrl($this->uriSigner->sign($this->urlGenerator->generate('cms_registration_activate', ['id' => $user->getId(), 'valid' => new \DateTimeImmutable('+3 days')->format('c')], UrlGeneratorInterface::ABSOLUTE_URL))),
            new Recipient($email, (string) $this->phone->format($user->getMobile()))
        );
    }
}
