<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Migration;

use Contao\CoreBundle\Migration\AbstractMigration;
use Contao\CoreBundle\Migration\MigrationResult;
use Doctrine\DBAL\Connection;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Host;
use Ferienpass\CoreBundle\Entity\User;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Uid\Uuid;

class UuidMigration extends AbstractMigration
{
    private readonly array $tables;

    public function __construct(
        private readonly Connection $db,
        #[Autowire(param: 'ferienpass.model.participant.class')]
        private readonly string $participantEntityClass,
        #[Autowire(param: 'ferienpass.model.offer.class')]
        private readonly string $offerEntityClass
    ) {
        $this->tables = [
            'Attendance' => Attendance::class,
            'Host' => Host::class,
            'Offer' => $this->offerEntityClass,
            'Participant' => $this->participantEntityClass,
            'User' => User::class,
        ];
    }

    public function shouldRun(): bool
    {
        $schemaManager = $this->db->createSchemaManager();

        foreach (array_keys($this->tables) as $tableName) {
            if (!$schemaManager->tablesExist([$tableName])) {
                continue;
            }

            $columns = $schemaManager->listTableColumns($tableName);
            if (!isset($columns['uuid'])) {
                return true;
            }
        }

        return false;
    }

    public function run(): MigrationResult
    {
        $schemaManager = $this->db->createSchemaManager();

        foreach (array_keys($this->tables) as $tableName) {
            if (!$schemaManager->tablesExist([$tableName])) {
                continue;
            }

            $columns = $schemaManager->listTableColumns($tableName);
            if (isset($columns['uuid'])) {
                continue;
            }

            $this->db->executeStatement("ALTER TABLE $tableName ADD COLUMN uuid BINARY(16) DEFAULT NULL");

            foreach ($this->db->executeQuery("SELECT id FROM $tableName WHERE uuid IS NULL")->fetchFirstColumn() as $id) {
                $uuid = Uuid::v4()->toRfc4122();
                $this->db->executeStatement("UPDATE $tableName SET uuid = UNHEX(REPLACE('$uuid', '-', '')) WHERE id = $id");
            }
        }

        return new MigrationResult(true, 'Migration successful');
    }
}
