<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\MessageHandler;

use Doctrine\ORM\EntityRepository;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Message\LotEdition as LotEditionCommand;
use Ferienpass\CoreBundle\Message\LotOffer;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Component\Lock\Key;
use Symfony\Component\Lock\LockFactory;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Messenger\Stamp\DispatchAfterCurrentBusStamp;

#[AsMessageHandler(bus: 'command.bus')]
class LotEdition
{
    /**
     * @param OfferRepositoryInterface&EntityRepository $offers
     */
    public function __construct(private readonly OfferRepositoryInterface $offers, private readonly MessageBusInterface $commandBus, private readonly LockFactory $lockFactory)
    {
    }

    public function __invoke(LotEditionCommand $command): void
    {
        $key = new Key('lotting.'.$command->getEditionId());
        $lock = $this->lockFactory->createLockFromKey($key, ttl: 15, autoRelease: false);
        $lock->acquire();

        $cancelled = OfferInterface::STATE_CANCELLED;

        /** @var array<array> $result */
        $result = $this->offers->createQueryBuilder('o')
            ->andWhere('o.edition = :edition_id')
            ->setParameter('edition_id', $command->getEditionId())
            ->andWhere('o.requiresApplication = 1')
            ->andWhere('o.onlineApplication = 1')
            ->andWhere("JSON_CONTAINS_PATH(o.status, 'one', '$.$cancelled') = 0")
            ->innerJoin('o.attendances', 'a')
            ->addSelect('sum(CASE WHEN a.status = :status_confirmed THEN 1 ELSE 0 END) AS countConfirmed')
            ->addSelect('sum(CASE WHEN a.status = :status_waiting THEN 1 ELSE 0 END) AS countWaiting')
            ->addSelect('sum(CASE WHEN a.status = :status_waiting OR a.status = :status_confirmed THEN 1 ELSE 0 END) AS countConfirmedOrWaiting')
            ->setParameter('status_confirmed', Attendance::STATUS_CONFIRMED)
            ->setParameter('status_waiting', Attendance::STATUS_WAITING)
            ->groupBy('o')
            ->orderBy('countWaiting', 'DESC')
            ->getQuery()
            ->getResult()
        ;

        // Process all offers with no waitlist
        foreach (array_filter($result, fn ($r) => null === $r[0]->getMaxParticipants() || $r['countConfirmedOrWaiting'] <= $r[0]->getMaxParticipants()) as $r) {
            /** @var OfferInterface $offer */
            $offer = $r[0];

            if (!$command->shouldIgnoreMinParticipants() && $r['countConfirmedOrWaiting'] < $offer->getMinParticipants()) {
                // $offer->setStateManually(OfferInterface::STATE_CANCELLED);
                continue;
            }

            foreach ($offer->getAttendancesWaiting() as $attendance) {
                if (!$attendance->isAllAlternateDates()
                    && !$attendance
                        ->getParticipant()
                        ->getAttendances()
                        ->filter(fn (Attendance $a) => $a->getOffer()->isVariantOf($attendance->getOffer()) && $a->isConfirmed())
                        ->isEmpty()) {
                    $attendance->setStatus(Attendance::STATUS_WITHDRAWN);
                    continue;
                }

                $attendance->setStatus(Attendance::STATUS_CONFIRMED);
            }
        }

        // Start assigning participants to offers
        // Begin with the most crowded offers (see @orderBy)
        // Luck and priority win for the crowded offers
        // After that, the algorithm aligns for "bad luck"
        $withOverhang = array_filter($result, fn ($r) => null !== $r[0]->getMaxParticipants() && $r['countConfirmedOrWaiting'] > $r[0]->getMaxParticipants() && $r['countWaiting'] > 0);
        foreach ($withOverhang as $r) {
            /** @var OfferInterface $offer */
            $offer = $r[0];

            $event = new LotOffer($offer->getId(), $key, $command->shouldUseWaitlist());
            $this->commandBus->dispatch(
                new Envelope($event)
                    ->with(new DispatchAfterCurrentBusStamp())
            );
        }
    }
}
