<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\MessageHandler;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Entity\PmPaymentTransactionStatus;
use Ferienpass\CoreBundle\Message\FetchPmPaymentStatus as FetchPmPaymentStatusCommand;
use Ferienpass\CoreBundle\Repository\PaymentRepository;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Workflow\Exception\NotEnabledTransitionException;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

#[AsMessageHandler(bus: 'command.bus')]
class FetchPmPaymentStatus
{
    public function __construct(private readonly HttpClientInterface $pmPaymentClient, #[Autowire(env: 'PMPAYMENT_AGS')] private readonly string $ags, private readonly EntityManagerInterface $entityManager, private PaymentRepository $payments, private WorkflowInterface $paymentStateMachine)
    {
    }

    public function __invoke(FetchPmPaymentStatusCommand $command): void
    {
        $txid = $command->getTxid();
        if ('' === $txid || '0' === $txid) {
            return;
        }

        $response = $this->pmPaymentClient->request('GET', '/payment/status/{ags}/{txid}', [
            'vars' => [
                'ags' => $this->ags,
                'txid' => $txid,
            ],
        ]);

        $content = $response->toArray();
        $isSuccessful = 1 === (int) $content['status'];
        $isPending = -1 === (int) $content['status'];

        /** @var Payment|null $payment */
        $payment = $this->payments->findOneBy(['pmPaymentTransactionId' => $txid]);

        $this->upsertStatus($txid, $content, $payment);

        if (!$isPending && !$payment->isFinalized()) {
            try {
                $this->paymentStateMachine->apply($payment, $isSuccessful ? Payment::TRANSITION_PAY : Payment::TRANSITION_FAIL);
            } catch (NotEnabledTransitionException) {
            }
        }
    }

    public function upsertStatus(string $txid, array $content, ?Payment $payment): void
    {
        $repo = $this->entityManager->getRepository(PmPaymentTransactionStatus::class);
        /** @var PmPaymentTransactionStatus|null $entity */
        $entity = $repo->findOneBy(['txid' => $txid]);

        $status = $content['details']['status'] ?? null;
        $modifiedAt = $content['details']['timestamp'] ? new \DateTimeImmutable($content['details']['timestamp']) : null;
        $createdAt = $content['details']['transactions'][0]['timestamp'] ?? null ? new \DateTimeImmutable($content['details']['transactions'][0]['timestamp']) : null;

        if (null !== $entity) {
            $entity->setStatus($status);
            $entity->setDescription($content['details']['description'] ?? $content['description'] ?? null);
            $entity->setCreatedAt($createdAt);
            $entity->setModifiedAt($modifiedAt);
            $entity->setPayload($content);

            return;
        }

        $entity = new PmPaymentTransactionStatus(
            $txid,
            $status,
            $content['details']['description'] ?? $content['description'] ?? null,
            $createdAt,
            $modifiedAt,
            $content
        );

        $this->entityManager->persist($entity);
        $payment?->setPmPaymentTransactionStatus($entity);
    }
}
