<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\MessageHandler;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Message\CommitAttendances as CommitAttendancesCommand;
use Ferienpass\CoreBundle\Message\DispatchDecisionLetters as DispatchDecisionLettersCommand;
use Ferienpass\CoreBundle\Messenger\UuidStamp;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Messenger\Stamp\DispatchAfterCurrentBusStamp;
use Symfony\Component\Uid\Uuid;

#[AsMessageHandler(bus: 'command.bus')]
class CommitAttendances
{
    public function __construct(private readonly AttendanceRepository $attendances, private readonly EditionRepository $editions, private readonly MessageBusInterface $commandBus)
    {
    }

    public function __invoke(CommitAttendancesCommand $command, Uuid $uuid): void
    {
        $edition = $this->editions->find($command->getEditionId());
        if (!$edition instanceof Edition) {
            return;
        }

        $this->commitAllAttendances($command->getAttendanceIds());

        if ($command->shouldSendDecisions()) {
            $this->commandBus->dispatch(
                new Envelope(new DispatchDecisionLettersCommand($command->getEditionId(), $command->getAttendanceIds()))
                    ->with(new UuidStamp($uuid), new DispatchAfterCurrentBusStamp())
            );
        }

        // The allocation tasks ends with commit
        $allocationTask = $edition->getActiveTasks('allocation');
        if (!$allocationTask->isEmpty()) {
            foreach ($allocationTask as $task) {
                $task->setPeriodEnd(new \DateTimeImmutable());
            }
        }
    }

    private function commitAllAttendances(array $attendanceIds): void
    {
        $qb1 = $this->attendances->createQueryBuilder('a1');
        $sub = $qb1->select('a1.id')->innerJoin('a1.offer', 'o1')->where('o1.disableOnlinePayment = :disable_true');

        $this->attendances
            ->createQueryBuilder('attendance')
            ->update()
            ->set('attendance.payable', true)
            ->where('attendance.status = :status')
            ->setParameter('status', Attendance::STATUS_CONFIRMED)
            ->andWhere('attendance.id IN (:ids)')
            ->andWhere(\sprintf('attendance.id NOT IN (%s)', $sub->getDQL()))
            ->setParameter('disable_true', true)
            ->setParameter('ids', $attendanceIds)
            ->getQuery()
            ->execute()
        ;
    }
}
