<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Facade;

use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Participant\BaseParticipant;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

class EraseDataFacade
{
    public function __construct(private readonly Connection $connection, private readonly ParticipantRepositoryInterface $participants, private readonly EntityManagerInterface $doctrine, #[Autowire(param: 'ferienpass.erase_data_period')] private readonly string $deleteAfterInterval, #[Autowire(param: 'ferienpass.model.participant.class')] private readonly string $participantEntityClass)
    {
    }

    public function eraseData(): void
    {
        // Retain participant age for statistics
        $this->retainParticipantAge();

        // Delete all participants that have attendances on events with past holiday
        $this->deleteParticipants();

        // Delete parents that have no participants and haven't logged in since a while
        $this->deleteMembersWithNoParticipants();
    }

    public function expiredParticipants(): array
    {
        $qb = $this->doctrine->createQueryBuilder();

        $qb
            ->select('p.id, p.lastname')
            ->from($this->participantEntityClass, 'p')
            ->leftJoin('p.attendances', 'a')
            ->leftJoin('a.offer', 'f')
            ->groupBy('p.id')
        ;

        $orX = [
            $qb->expr()->andX(
                $qb->expr()->isNull('f'),
                $qb->expr()->orX(
                    $qb->expr()->isNull('p.createdAt'),
                    $qb->expr()->lt('p.createdAt', \sprintf('DATE_SUB(CURRENT_TIMESTAMP(), %s)', preg_replace('/^(\d+)\s+([A-Za-z]+)$/', '$1, \'$2\'', $this->deleteAfterInterval)))
                )
            ),
        ];

        $qb
            ->leftJoin('f.edition', 'e')->leftJoin('e.tasks', 'et')
            ->setParameter('showOffers', 'show_offers')
        ;

        $orX[] = $qb->expr()->andX(
            $qb->expr()->eq('et.type', ':showOffers'),
            $qb->expr()->lt('et.periodEnd', \sprintf('DATE_SUB(CURRENT_TIMESTAMP(), %s)', preg_replace('/^(\d+)\s+([A-Za-z]+)$/', '$1, \'$2\'', $this->deleteAfterInterval)))
        );

        $qb
            ->where($qb->expr()->orX(...$orX))
            ->orderBy('p.lastname', 'ASC')
        ;

        $participantsToDelete = $qb->getQuery()->getScalarResult();

        $qb = $this->doctrine->createQueryBuilder();

        $qb
            ->select('p.id')
            ->from($this->participantEntityClass, 'p')
            ->innerJoin('p.attendances', 'a')
            ->innerJoin('a.offer', 'f')
            ->leftJoin('f.dates', 'd')
            ->leftJoin('p.user', 'u')
            ->groupBy('p.id')
        ;

        $orX = [
            $qb->expr()->gt('d.end', 'CURRENT_TIMESTAMP()'),
            $qb->expr()->andX(
                $qb->expr()->isNotNull('u.dontDeleteBefore'),
                $qb->expr()->gt('u.dontDeleteBefore', 'CURRENT_TIMESTAMP()')
            ),
        ];

        $qb->innerJoin('f.edition', 'e')->innerJoin('e.tasks', 'et');

        $orX[] = $qb->expr()->gt('et.periodEnd', 'CURRENT_TIMESTAMP()');

        $qb->where($qb->expr()->orX(...$orX));

        $participantsToKeep = $qb->getQuery()->getScalarResult();

        $participantsToDelete = array_column($participantsToDelete, 0);
        $participantsToKeep = array_column($participantsToKeep, 0);

        $delete = array_diff($participantsToDelete, $participantsToKeep);
        if ($delete) {
            return $this->participants->findBy(['id' => array_diff($participantsToDelete, $participantsToKeep)]);
        }

        return [];
    }

    private function deleteParticipants(): void
    {
        $participants = $this->expiredParticipants();

        /** @var BaseParticipant $participant */
        foreach ($participants as $participant) {
            /** @var Attendance $attendance */
            $pseudonym = bin2hex(random_bytes(5));
            foreach ($participant->getAttendances() as $attendance) {
                // Create a pseudonym for each participant
                $attendance->setParticipantPseudonym($pseudonym);

                // Remove parent association so attendances do not get removed
                $attendance->unsetParticipant();
            }
        }

        $this->doctrine->flush();

        $this->participants
            ->createQueryBuilder('p')
            ->delete()
            ->where('p IN (:ids)')
            ->setParameter('ids', $participants)
            ->getQuery()
            ->execute()
        ;
    }

    private function deleteMembersWithNoParticipants(): void
    {
        $userIds = $this->doctrine->getRepository(User::class)
            ->createQueryBuilder('u')
            ->select('u.id')
            ->leftJoin('u.participants', 'p')
            ->where('p.id IS NULL')
            // ->andWhere('u.lastLogin < DATE_SUB(NOW(), INTERVAL 2 WEEK)')
            ->andWhere("JSON_SEARCH(u.roles, 'one', :role_member) IS NOT NULL")
            ->andWhere("JSON_SEARCH(u.roles, 'one', :role_host) IS NULL")
            ->andWhere("JSON_SEARCH(u.roles, 'one', :role_admin) IS NULL")
            ->andWhere("JSON_SEARCH(u.roles, 'one', :role_sadmin) IS NULL")
            ->andWhere('u.dontDeleteBefore IS NULL OR u.dontDeleteBefore < CURRENT_TIMESTAMP()')
            ->setParameter('role_member', 'ROLE_MEMBER')
            ->setParameter('role_host', 'ROLE_HOST')
            ->setParameter('role_admin', 'ROLE_ADMIN')
            ->setParameter('role_sadmin', 'ROLE_SUPER_ADMIN')
            ->getQuery()
            ->getSingleColumnResult()
        ;

        $this->doctrine->getRepository(User::class)
            ->createQueryBuilder('u')
            ->delete()
            ->where('u.id IN (:ids)')
            ->setParameter('ids', $userIds, ArrayParameterType::INTEGER)
            ->getQuery()
            ->execute()
        ;
    }

    private function retainParticipantAge(): void
    {
        $this->connection->executeQuery(
            <<<'SQL'
                UPDATE Attendance a
                INNER JOIN Participant p ON a.participant_id = p.id
                INNER JOIN Offer f ON a.offer_id = f.id
                LEFT OUTER JOIN OfferDate d ON d.offer_id = f.id
                SET age = (IF((p.dateOfBirth IS NULL), null, TIMESTAMPDIFF(YEAR, p.dateOfBirth, d.begin)))
                WHERE a.age IS NULL
                SQL
        )->rowCount();
    }
}
