<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Facade;

use Doctrine\Common\Collections\Criteria;
use Doctrine\Common\Collections\Order;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\ApplicationSystem\ApplicationSystems;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Debtor;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Message\ReorderParticipantList;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Workflow\WorkflowInterface;

class AttendanceFacade
{
    public function __construct(private readonly MessageBusInterface $commandBus, private readonly EntityManagerInterface $entityManager, private readonly ApplicationSystems $applicationSystems, private readonly AttendanceRepository $attendances, private readonly WorkflowInterface $attendanceStateMachine)
    {
    }

    /**
     * Preview an attendance status without persisting the attendance.
     */
    public function preview(OfferInterface $offer, ParticipantInterface $participant): Attendance
    {
        $attendance = new Attendance($offer, $participant);

        $applicationSystem = $this->applicationSystems->findApplicationSystem($offer);
        $applicationSystem?->suggestTransition($attendance);

        return $attendance;
    }

    /**
     * Create an attendance (or update an existing record) for a given participant and offer.
     *
     * If an explicit status is given, no application procedure is used.
     * Setting an explicit attendance status shall only be possible for admins.
     *
     * @throws \RuntimeException in case no unambiguous application system is applicable
     */
    public function create(OfferInterface $offer, ParticipantInterface $participant, ?string $status = null, bool $notify = true, bool $allAlternateDates = true, $commit = true): ?Attendance
    {
        $attendance = $this->findOrCreateAttendance($offer, $participant, $allAlternateDates);

        // Create debtor for given user
        if (($debtor = $participant->getUser()?->getDebtor()) instanceof Debtor) {
            $this->entityManager->persist($debtor);
        }

        // Reset status to allow re-assignment from current application system
        if (null !== $status) {
            $attendance->resetStatus();
        }

        // We need the entity persisted because we will dispatch messages
        if (!$attendance->getId()) {
            $this->entityManager->flush();
        }

        if (null === $attendance->getStatus() || $attendance->isHidden()) {
            $attendance->setStatus(Attendance::STATUS_WAITING);
        }

        $initialTransition = match ($status) {
            Attendance::STATUS_CONFIRMED => Attendance::TRANSITION_CONFIRM,
            Attendance::STATUS_WAITLISTED => Attendance::TRANSITION_WAITLIST,
            default => null,
        };

        if (null !== $initialTransition) {
            $this->attendanceStateMachine->apply($attendance, $initialTransition, [
                'initial' => true,
                'notify' => $notify,
                'commit' => $commit,
            ]);
        } else {
            $applicationSystem = $this->applicationSystems->findApplicationSystem($attendance->getOffer());
            $transition = $applicationSystem?->suggestTransition($attendance);

            if ($transition) {
                $this->attendanceStateMachine->apply($attendance, $transition, [
                    'initial' => true,
                    'notify' => $notify,
                    'commit' => $commit,
                ]);
            }
        }

        $this->entityManager->flush();

        $this->commandBus->dispatch(new ReorderParticipantList($offer->getId()));

        return $attendance;
    }

    public function increasePriority(Attendance $attendance): void
    {
        // The priority only applies for non-assigned spots
        if (!$attendance->isWaiting()) {
            return;
        }

        $attendances = $attendance->getParticipant()
            ?->getAttendancesWaiting()
            ?->matching(Criteria::create()->orderBy(['user_priority' => Order::Ascending]))
        ;

        if (null === $attendances || $attendances->isEmpty()) {
            $attendance->setUserPriority(1);
            $this->entityManager->flush();

            return;
        }

        foreach ($attendances as $currentAttendance) {
            if ($attendance === $currentAttendance) {
                $attendance->setUserPriority(min(1, $attendance->getUserPriority() - 1));
                continue;
            }

            $currentAttendance->setUserPriority($currentAttendance->getUserPriority() + 1);
        }

        $this->entityManager->flush();
    }

    private function findOrCreateAttendance(OfferInterface $offer, ParticipantInterface $participant, bool $allAlternateDates): Attendance
    {
        $attendance = $this->attendances->findExisting($offer, $participant);
        if (!$attendance instanceof Attendance) {
            $attendance = new Attendance($offer, $participant);
            $attendance->setAllAlternateDates($allAlternateDates);
            $offer->addAttendance($attendance);

            return $attendance;
        }

        //        // When an attendance already exists for this user, this action is immutable
        //        // But if existing attendance was withdrawn from the user, allow re-sign-ups
        //        if (!$attendance->isHidden()) {
        //            return null;
        //        }

        $attendance->setAllAlternateDates($allAlternateDates);

        return $attendance;
    }
}
