<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Export\ParticipantList;

use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx as XlsxWriter;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Filesystem\Filesystem;

final readonly class XlsxExport
{
    public function __construct(
        private Filesystem $filesystem,
        #[Autowire(param: 'ferienpass.participant_list_sort')]
        private string $sort
    ) {
    }

    public function generate(OfferInterface $offer, ?string $destination = null): string
    {
        $spreadsheet = $this->createSpreadsheet($offer);
        $hash = md5(serialize($offer->getId()));
        $xlsxPath = \sprintf('%s/xlsx-%s.xlsx', sys_get_temp_dir(), $hash);

        $writer = new XlsxWriter($spreadsheet);
        $writer->save($xlsxPath);

        if (null !== $destination) {
            $this->filesystem->copy($xlsxPath, $destination);
        }

        return $destination ?? $xlsxPath;
    }

    private function createSpreadsheet(OfferInterface $offer): Spreadsheet
    {
        $spreadsheet = new Spreadsheet();
        $spreadsheet->getDefaultStyle()->getFont()->setName('Arial');
        $sheet = $spreadsheet->getActiveSheet();

        // Set title
        $sheet->setTitle('Teilnehmerliste');

        // Header
        $sheet->setCellValue('A1', 'Teilnehmerliste: '.$offer->getName());
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        $sheet->mergeCells('A1:F1');

        // Column headers
        $headers = ['Nr.', 'Name', 'Vorname', 'Geburtsdatum', 'Alter', 'Status'];
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col.'3', $header);
            $sheet->getStyle($col.'3')->getFont()->setBold(true);
            ++$col;
        }

        // Get participants
        $participants = $offer->getAttendancesConfirmed();

        // Sort participants based on configuration
        $participantsArray = iterator_to_array($participants);
        if ('name' === $this->sort) {
            usort($participantsArray, function ($a, $b) {
                $nameA = $a->getParticipant()->getLastname().' '.$a->getParticipant()->getFirstname();
                $nameB = $b->getParticipant()->getLastname().' '.$b->getParticipant()->getFirstname();

                return strcasecmp($nameA, $nameB);
            });
        }

        // Data rows
        $row = 4;
        $nr = 1;
        foreach ($participantsArray as $attendance) {
            $participant = $attendance->getParticipant();

            $sheet->setCellValue('A'.$row, $nr);
            $sheet->setCellValue('B'.$row, $participant->getLastname());
            $sheet->setCellValue('C'.$row, $participant->getFirstname());

            if ($participant->getDateOfBirth()) {
                $sheet->setCellValue('D'.$row, $participant->getDateOfBirth()->format('d.m.Y'));
                $age = $participant->getDateOfBirth()->diff(new \DateTime())->y;
                $sheet->setCellValue('E'.$row, $age);
            }

            $sheet->setCellValue('F'.$row, $attendance->getStatus());

            ++$row;
            ++$nr;
        }

        // Auto-size columns
        foreach (range('A', 'F') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        return $spreadsheet;
    }
}
