<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Export\ParticipantList;

use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Mpdf\Mpdf;
use Mpdf\Output\Destination as MpdfDestination;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Twig\Environment as TwigEnvironment;

final readonly class PdfExport
{
    public function __construct(
        private Filesystem $filesystem,
        private TwigEnvironment $twig,
        private EventDispatcherInterface $dispatcher,
        #[Autowire(param: 'ferienpass.participant_list_sort')]
        private string $sort,
        #[Autowire(param: 'kernel.cache_dir')]
        private string $cacheDir
    ) {
    }

    public function generate(OfferInterface $offer, ?string $destination = null): string
    {
        $html = $this->renderHtml($offer);
        $hash = md5($html);
        $pdfPath = \sprintf('%s/pdf-%s.pdf', sys_get_temp_dir(), $hash);

        $this->createPdf($pdfPath, $html);

        if (null !== $destination) {
            $this->filesystem->copy($pdfPath, $destination);
        }

        return $destination ?? $pdfPath;
    }

    private function renderHtml(OfferInterface $offer): string
    {
        return $this->twig->render('@FerienpassCore/participant_list/pdf/page.html.twig', [
            'offer' => $offer,
            'dispatcher' => $this->dispatcher,
            'sort' => $this->sort,
        ]);
    }

    private function createPdf(string $path, string $html, array $mPdfConfig = []): void
    {
        if (file_exists($path)) {
            return;
        }

        $mPdfConfig = array_merge([
            'tempDir' => $this->cacheDir,
            'default_font_size' => 12,
            'default_font' => 'roboto',
            'fontDir' => \dirname(__DIR__).'/../../fonts/Roboto',
            'fontdata' => [
                'roboto' => [
                    'R' => 'Roboto-Regular.ttf',
                    'B' => 'Roboto-Bold.ttf',
                    'I' => 'Roboto-Italic.ttf',
                ],
                'roboto-light' => [
                    'R' => 'Roboto-Light.ttf',
                    'I' => 'Roboto-LightItalic.ttf',
                ],
                'roboto-thin' => [
                    'R' => 'Roboto-Thin.ttf',
                    'I' => 'Roboto-ThinItalic.ttf',
                ],
            ],
        ], $mPdfConfig);

        $mPdf = new Mpdf($mPdfConfig);

        $mPdf->WriteHTML($html);

        $mPdf->Output($path, MpdfDestination::FILE);
    }
}
