<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Export\Offer\PrintSheet;

use Contao\CoreBundle\Filesystem\VirtualFilesystemInterface;
use Ferienpass\CoreBundle\Export\Offer\OffersExportInterface;
use Mpdf\Mpdf;
use Mpdf\Output\Destination as MpdfDestination;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\Exclude;
use Twig\Environment;

#[Exclude]
class PdfExport implements OffersExportInterface
{
    public function __construct(private readonly array $config, private readonly string $template, private readonly VirtualFilesystemInterface $exportStorage, private readonly Environment $twig, #[Autowire(param: 'kernel.project_dir')] private readonly string $projectDir, #[Autowire(param: 'kernel.cache_dir')] private string $cacheDir)
    {
    }

    public static function getName(): string
    {
        // Managed via DI extension
        return '';
    }

    public function generate(iterable $offers, ?string $destination = null, bool $stream = false): mixed
    {
        ini_set('pcre.backtrack_limit', '100000000');

        $html = $this->render($offers);
        $hash = md5($html);

        $destination ??= \sprintf('pdf-%s.pdf', $hash);
        $path = 'pdf/'.$destination;

        $content = $this->createPdf($html, $this->config);
        if ($stream) {
            $stream = fopen('php://temp', 'r+');
            fwrite($stream, $content);
            rewind($stream);

            return $stream;
        }

        $this->exportStorage->write($path, $content);

        return $path;
    }

    private function render(iterable $items): string
    {
        return $this->twig->render($this->template, [
            'items' => $items,
        ]);
    }

    private function createPdf(string $html, array $mPdfConfig): string
    {
        $mPdfConfig = array_merge(['tempDir' => $this->cacheDir], $mPdfConfig);

        $mPdf = new Mpdf($mPdfConfig);
        $mPdf->SetBasePath($this->projectDir);

        // disable errors for now (undefined array index inside mpdf)
        @$mPdf->WriteHTML($html);

        return $mPdf->Output(dest: MpdfDestination::STRING_RETURN);
    }
}
