<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\EventListener\Workflow;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Message\WithdrawAttendance;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Workflow\Attribute\AsEnteredListener;
use Symfony\Component\Workflow\Event\EnteredEvent;

#[AsEnteredListener(workflow: 'attendance', place: Attendance::STATUS_WITHDRAWN)]
class AttendanceWithdrawTransactionListener
{
    public function __construct(private readonly MessageBusInterface $commandBus, #[Autowire(param: 'ferienpass.withdraw_grace_period')] private readonly ?string $gracePeriod)
    {
    }

    public function __invoke(EnteredEvent $event): void
    {
        $attendance = $event->getSubject();
        if (!$attendance instanceof Attendance) {
            throw new \RuntimeException('Unexpected event subject');
        }

        $context = $event->getContext();
        $shallReorder = $context['reorder'] ?? true;
        $shallNotify = $context['notify'] ?? true;

        if ($this->isInGracePeriod($attendance)) {
            $attendance->setPayable(false);
        }

        $command = new WithdrawAttendance($attendance->getId(), $shallReorder, $shallNotify);
        $this->commandBus->dispatch($command);
    }

    private function isInGracePeriod(Attendance $attendance): bool
    {
        $gracePeriod = $this->gracePeriod ? new \DateTimeImmutable($this->gracePeriod) : null;

        return !$gracePeriod instanceof \DateTimeImmutable || ($attendance->getOffer()->getDates()->isEmpty() || $attendance->getOffer()->getDates()->first()->getBegin() > $gracePeriod);
    }
}
