<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Participant\BaseParticipant;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Misd\PhoneNumberBundle\Validator\Constraints\PhoneNumber;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Serializer\Attribute\Groups;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;

#[ORM\Entity(repositoryClass: UserRepository::class)]
#[UniqueEntity('email')]
class User implements UserInterface, PasswordAuthenticatedUserInterface
{
    public const ACCOUNT_ROLES = [
        'ROLE_MEMBER',
        'ROLE_HOST',
        'ROLE_ADMIN',
    ];

    public const ADMIN_ROLES = [
        'ROLE_PARTICIPANTS_ADMIN',
        'ROLE_PAYMENTS_ADMIN',
        'ROLE_CMS_ADMIN',
    ];

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: UuidType::NAME)]
    private Uuid $uuid;

    #[Assert\Email]
    #[Assert\NotBlank]
    #[ORM\Column(type: 'string', length: 180, unique: true)]
    #[Groups(['notification', 'admin_list'])]
    private ?string $email = null;

    #[ORM\Column(type: 'string', nullable: true)]
    #[Groups(['notification', 'admin_list'])]
    #[Assert\NotBlank]
    private ?string $firstname = null;

    #[ORM\Column(type: 'string', nullable: true)]
    #[Groups(['notification', 'admin_list'])]
    #[Assert\NotBlank]
    private ?string $lastname = null;

    #[ORM\Column(type: 'string', length: 64, nullable: true)]
    #[Groups(['notification', 'admin_list'])]
    #[PhoneNumber(type: PhoneNumber::FIXED_LINE, defaultRegion: 'DE')]
    private ?string $phone = null;

    #[ORM\Column(type: 'string', length: 64, nullable: true)]
    #[Groups(['notification', 'admin_list'])]
    #[PhoneNumber(type: PhoneNumber::MOBILE, defaultRegion: 'DE')]
    private ?string $mobile = null;

    #[ORM\JoinTable(name: 'UserAddresses')]
    #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
    #[ORM\InverseJoinColumn(name: 'address_id', referencedColumnName: 'id', unique: true)]
    #[ORM\ManyToMany(targetEntity: PostalAddress::class, cascade: ['persist'])]
    private Collection $postalAddresses;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    #[Groups(['notification', 'admin_list'])]
    private \DateTimeInterface $createdAt;

    #[ORM\Column(type: 'datetime', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $modifiedAt;

    #[ORM\Column(type: 'datetime', nullable: true)]
    private ?\DateTimeInterface $lastLogin = null;

    #[ORM\Column(type: 'datetime', nullable: true)]
    private ?\DateTimeInterface $dontDeleteBefore = null;

    #[ORM\Column(type: 'json')]
    private array $roles = [];

    #[ORM\Column(type: 'simple_array', nullable: true)]
    private ?array $publicFields = [];

    #[ORM\Column(type: 'string', nullable: true)]
    private ?string $password = null;

    #[Assert\Length(max: 4096)]
    private ?string $plainPassword = null;

    #[ORM\Column(type: 'boolean')]
    private bool $disable = false;

    #[ORM\Column(type: 'boolean')]
    private bool $blockedEmail = false;

    #[ORM\OneToOne(mappedBy: 'user', targetEntity: BlockedContact::class, cascade: ['persist'])]
    private ?BlockedContact $blockedContact = null;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: HostMemberAssociation::class, cascade: ['persist', 'remove'], orphanRemoval: true)]
    private Collection $hostAssociations;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: ParticipantInterface::class, cascade: ['persist', 'remove'], fetch: 'EXTRA_LAZY')]
    private Collection $participants;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Payment::class)]
    private Collection $payments;

    #[ORM\OneToOne(mappedBy: 'user', targetEntity: Debtor::class, cascade: ['persist'], fetch: 'EAGER')]
    private ?Debtor $debtor = null;

    #[ORM\OneToMany(mappedBy: 'user', targetEntity: Consent::class)]
    private Collection $consents;

    #[ORM\OneToMany(mappedBy: 'ownedBy', targetEntity: ApiToken::class)]
    private Collection $apiTokens;

    #[ORM\OneToMany(mappedBy: 'account', targetEntity: UserActivity::class, cascade: ['persist', 'remove'], orphanRemoval: true)]
    #[ORM\OrderBy(['createdAt' => 'DESC'])]
    private Collection $activity;

    /* Scopes given during API authentication */
    private ?array $accessTokenScopes = null;

    public function __construct()
    {
        $this->uuid = Uuid::v4();
        $this->createdAt = new \DateTimeImmutable();
        $this->modifiedAt = new \DateTimeImmutable();
        $this->postalAddresses = new ArrayCollection();
        $this->hostAssociations = new ArrayCollection();
        $this->participants = new ArrayCollection();
        $this->payments = new ArrayCollection();
        $this->consents = new ArrayCollection();
        $this->apiTokens = new ArrayCollection();
        $this->activity = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getUuid(): Uuid
    {
        return $this->uuid;
    }

    public function getFirstname(): ?string
    {
        return $this->firstname;
    }

    public function setFirstname(?string $firstname): void
    {
        $this->firstname = $firstname;
    }

    public function getLastname(): ?string
    {
        return $this->lastname;
    }

    public function getName(): string
    {
        return \sprintf('%s %s', $this->getFirstname(), $this->getLastname());
    }

    public function setLastname(?string $lastname): void
    {
        $this->lastname = $lastname;
    }

    public function getPhone(): ?string
    {
        return $this->phone;
    }

    public function setPhone(?string $phone): void
    {
        $this->phone = $phone;
    }

    public function getMobile(): ?string
    {
        return $this->mobile;
    }

    public function setMobile(?string $mobile): void
    {
        $this->mobile = $mobile;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function getModifiedAt(): \DateTimeInterface
    {
        return $this->modifiedAt;
    }

    public function setModifiedAt(\DateTimeInterface $dateTime = new \DateTime()): void
    {
        $this->modifiedAt = $dateTime;
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(?string $email): self
    {
        $this->email = $email;

        return $this;
    }

    public function getUserIdentifier(): string
    {
        return (string) $this->email;
    }

    public function getUsername(): string
    {
        return $this->getUserIdentifier();
    }

    public function getRoles(): array
    {
        $roles[] = $this->roles;

        if (null !== $this->accessTokenScopes) {
            $roles[] = $this->accessTokenScopes;
        }

        // guarantee every user at least has ROLE_USER
        $roles[] = ['ROLE_USER'];

        return array_unique(array_merge(...$roles));
    }

    public function setRoles(array $roles): self
    {
        $this->roles = $roles;

        return $this;
    }

    public function addAdminRole(string $role): void
    {
        if (!\in_array($role, self::ADMIN_ROLES, true) || \in_array($role, $this->roles, true)) {
            return;
        }

        $this->roles[] = $role;
    }

    public function removeAdminRole(string $role): void
    {
        if (!\in_array($role, self::ADMIN_ROLES, true) || !\in_array($role, $this->roles, true)) {
            return;
        }

        foreach (array_keys($this->roles, $role, true) as $key) {
            unset($this->roles[$key]);
        }

        $this->roles = array_values($this->roles);
    }

    public function getAdminRoles(): array
    {
        return array_intersect($this->roles, self::ADMIN_ROLES);
    }

    public function addAccountRole(string $role): void
    {
        if (!\in_array($role, self::ACCOUNT_ROLES, true) || \in_array($role, $this->roles, true)) {
            return;
        }

        $this->roles[] = $role;
    }

    public function removeAccountRole(string $role): void
    {
        if (!\in_array($role, self::ACCOUNT_ROLES, true) || !\in_array($role, $this->roles, true)) {
            return;
        }

        foreach (array_keys($this->roles, $role, true) as $key) {
            unset($this->roles[$key]);
        }

        $this->roles = array_values($this->roles);
    }

    public function getAccountRoles(): array
    {
        return array_intersect($this->roles, self::ACCOUNT_ROLES);
    }

    public function getPassword(): ?string
    {
        return $this->password;
    }

    public function setPassword(string $password): self
    {
        $this->password = $password;

        return $this;
    }

    public function getPlainPassword(): ?string
    {
        return $this->plainPassword;
    }

    public function setPlainPassword(?string $plainPassword): void
    {
        $this->plainPassword = $plainPassword;
    }

    #[Groups(['admin_list'])]
    public function isDisabled(): bool
    {
        return $this->disable;
    }

    public function setDisabled(bool $disable = true): void
    {
        $this->disable = $disable;
    }

    public function isBlockedEmail(): bool
    {
        return $this->blockedEmail;
    }

    public function setBlockedEmail(bool $blockedEmail): void
    {
        $this->blockedEmail = $blockedEmail;
    }

    public function getBlockedContact(): ?BlockedContact
    {
        return $this->blockedContact;
    }

    public function isSuperAdmin(): bool
    {
        return \in_array('ROLE_SUPER_ADMIN', $this->getRoles(), true);
    }

    public function isAdmin(): bool
    {
        return \in_array('ROLE_ADMIN', $this->getRoles(), true);
    }

    public function isHost(): bool
    {
        return \in_array('ROLE_HOST', $this->getRoles(), true);
    }

    public function isMember(): bool
    {
        return \in_array('ROLE_MEMBER', $this->getRoles(), true);
    }

    public function setSuperAdmin(bool $superAdmin): void
    {
        if ($superAdmin) {
            if (\in_array('ROLE_SUPER_ADMIN', $this->roles, true)) {
                return;
            }

            $this->roles[] = 'ROLE_SUPER_ADMIN';

            return;
        }

        if (!\in_array('ROLE_SUPER_ADMIN', $this->roles, true)) {
            return;
        }

        foreach (array_keys($this->roles, 'ROLE_SUPER_ADMIN', true) as $key) {
            unset($this->roles[$key]);
        }

        // Make sure the ROLE_ADMIN stays
        if (!\in_array('ROLE_ADMIN', $this->roles, true)) {
            $this->roles[] = 'ROLE_ADMIN';
        }

        $this->roles = array_values($this->roles);
    }

    public function getPostalAddresses(): Collection
    {
        return $this->postalAddresses;
    }

    public function addPostalAddress(?PostalAddress $address): void
    {
        if (!$address instanceof PostalAddress) {
            return;
        }

        $this->postalAddresses->add($address);
    }

    public function removePostalAddress(PostalAddress $address): void
    {
        if (!$this->postalAddresses->contains($address)) {
            return;
        }

        $this->postalAddresses->removeElement($address);
    }

    public function getHosts(): Collection
    {
        return $this->hostAssociations->map(fn (HostMemberAssociation $a) => $a->getHost());
    }

    public function hasHost(Host $host): bool
    {
        return !$this->hostAssociations
            ->matching(Criteria::create()->where(Criteria::expr()->eq('host', $host)))
            ->isEmpty();
    }

    public function addHost(Host $host)
    {
        if ($this->hasHost($host)) {
            return;
        }

        $this->hostAssociations[] = new HostMemberAssociation($this, $host);
    }

    public function removeHost(Host $host)
    {
        /** @var HostMemberAssociation $hostAssociation */
        foreach ($this->hostAssociations as $hostAssociation) {
            if ($hostAssociation->getHost() === $host) {
                $this->hostAssociations->removeElement($hostAssociation);

                return;
            }
        }
    }

    public function getLastLogin(): ?\DateTimeInterface
    {
        return $this->lastLogin;
    }

    public function setLastLogin(?\DateTimeInterface $lastLogin): void
    {
        $this->lastLogin = $lastLogin;
    }

    public function getPublicFields(): ?array
    {
        return $this->publicFields;
    }

    public function setPublicFields(array $publicFields): void
    {
        $this->publicFields = $publicFields;
    }

    public function getDontDeleteBefore(): ?\DateTimeInterface
    {
        return $this->dontDeleteBefore;
    }

    public function setDontDeleteBefore(?\DateTimeInterface $dontDeleteBefore): void
    {
        $this->dontDeleteBefore = $dontDeleteBefore;
    }

    /** @return Collection<int, BaseParticipant> */
    public function getParticipants(): Collection
    {
        return $this->participants;
    }

    public function addParticipant(ParticipantInterface $participant): void
    {
        $participant->setUser($this);

        $this->participants->add($participant);
    }

    public function removeParticipant(?ParticipantInterface $participant = null): void
    {
        if (!$this->participants->contains($participant)) {
            return;
        }

        $this->participants->removeElement($participant);
    }

    public function eraseCredentials(): void
    {
        $this->plainPassword = null;
    }

    public function getPayments(): Collection
    {
        return $this->payments;
    }

    public function getConsents(): Collection
    {
        return $this->consents;
    }

    public function getActivity(): Collection
    {
        return $this->activity;
    }

    public function getComments(): Collection
    {
        return $this->activity->filter(fn (UserActivity $l) => $l->isComment());
    }

    public function getApiTokens(): Collection
    {
        return $this->apiTokens;
    }

    public function markAsTokenAuthenticated(array $scopes): void
    {
        $this->accessTokenScopes = $scopes;
    }

    public function getDebtor(): Debtor
    {
        if (!$this->debtor instanceof Debtor) {
            $this->createDebtor();
        }

        return $this->debtor;
    }

    public function createDebtor(): void
    {
        if ($this->debtor instanceof Debtor) {
            return;
        }

        $this->debtor = new Debtor($this);
    }
}
