<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\ORM\Mapping as ORM;

#[ORM\Entity]
class PaymentItem
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\ManyToOne(targetEntity: Attendance::class, inversedBy: 'paymentItems')]
    #[ORM\JoinColumn(onDelete: 'SET NULL')]
    private ?Attendance $attendance;

    #[ORM\Column(type: 'integer', options: ['unsigned' => false])]
    private int $amount;

    #[ORM\Column(type: 'string', length: 64, nullable: false)]
    private string $type;

    public function __construct(
        #[ORM\ManyToOne(targetEntity: Payment::class, fetch: 'EAGER', inversedBy: 'items')]
        private ?Payment $payment,
        int $amount,
        ?Attendance $attendance = null,
        string $type = 'attendance'
    ) {
        if ('attendance' === $type && !$attendance instanceof Attendance) {
            throw new \InvalidArgumentException('Attendance object is missing');
        }

        if ('discount' === $type && $amount > 0) {
            throw new \InvalidArgumentException('Discounts should be negative');
        }

        $this->createdAt = new \DateTimeImmutable();
        $this->amount = $amount;
        $this->attendance = $attendance;
        $this->type = $type;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getPayment(): Payment
    {
        return $this->payment;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function getAttendance(): ?Attendance
    {
        return $this->attendance;
    }

    public function getAmount(): int
    {
        return $this->amount;
    }

    public function setAmount(int $amount): void
    {
        $this->amount = $amount;
    }

    public function removeAttendanceAssociation(): void
    {
        $this->attendance = null;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function isAttendance(): bool
    {
        return 'attendance' === $this->type;
    }

    public function isDiscount(): bool
    {
        return 'discount' === $this->type;
    }
}
