<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Dto\Currency;
use Ferienpass\CoreBundle\Repository\PaymentRepository;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

#[ORM\Entity(repositoryClass: PaymentRepository::class)]
class Payment
{
    final public const string STATUS_CREATED = 'created';
    final public const string TRANSITION_ABANDON = 'abandon';
    final public const string STATUS_ABANDONED = 'abandoned';
    final public const string TRANSITION_PAY = 'pay';
    final public const string STATUS_PAID = 'paid';
    final public const string TRANSITION_FAIL = 'fail';
    final public const string STATUS_FAILED = 'failed';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: UuidType::NAME)]
    private Uuid $uuid;

    #[Groups('admin_list')]
    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\OneToMany(mappedBy: 'payment', targetEntity: PaymentItem::class, cascade: ['persist'], orphanRemoval: true)]
    private Collection $items;

    #[ORM\Column(type: 'integer', options: ['unsigned' => false])]
    private int $totalAmount = 0;

    #[Groups('admin_list')]
    #[ORM\Column(type: 'text', nullable: true)]
    private ?string $billingAddress = null;

    #[Groups('admin_list')]
    #[ORM\Column(type: 'text', length: 255, nullable: true)]
    private ?string $billingEmail = null;

    #[ORM\OneToMany(mappedBy: 'payment', targetEntity: BookEntry::class, cascade: ['persist'])]
    private Collection $bookEntries;

    #[ORM\Column(name: 'pmpayment_txid', type: 'string', length: 64, nullable: true)]
    private ?string $pmPaymentTransactionId = null;

    #[ORM\OneToOne(targetEntity: PmPaymentTransactionStatus::class)]
    #[ORM\JoinColumn(name: 'pmpayment_status', referencedColumnName: 'id')]
    private ?PmPaymentTransactionStatus $pmPaymentTransactionStatus = null;

    public function __construct(#[ORM\ManyToOne(targetEntity: Debtor::class, cascade: ['persist'], inversedBy: 'payments')]
        private ?Debtor $debtor, #[Groups('admin_list')]
        #[ORM\Column(type: 'text', length: 255, nullable: true)]
        private ?string $receiptNumber = null, #[ORM\ManyToOne(targetEntity: User::class, inversedBy: 'payments')]
        #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
        private ?User $user = null, #[Groups('admin_list')]
        #[ORM\Column(type: 'text', length: 64, nullable: true)]
        private string $status = self::STATUS_CREATED, ?Uuid $uuid = null)
    {
        $this->uuid = $uuid ?? Uuid::v4();
        $this->createdAt = new \DateTimeImmutable();
        $this->items = new ArrayCollection();
        $this->bookEntries = new ArrayCollection();

        $this->populateAddressFromUser($this->debtor->getUser());
    }

    /**
     * @param array<Attendance> $attendances
     */
    public static function fromAttendances(Debtor $debtor, iterable $attendances, EventDispatcherInterface $dispatcher, ?string $receiptNumber = null, ?User $user = null): static
    {
        $self = new self($debtor, $receiptNumber, user: $user);

        foreach ($attendances as $attendance) {
            $self->items->add(new PaymentItem(
                $self,
                $attendance->getOffer()->getFeePayable($attendance->getParticipant(), $dispatcher),
                $attendance,
            ));
        }

        $self->calculateTotalAmount();

        return $self;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getUuid(): Uuid
    {
        return $this->uuid;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    /** @return Collection<PaymentItem> */
    public function getItems(): Collection
    {
        return $this->items;
    }

    public function addItem(PaymentItem $item): void
    {
        $this->items->add($item);

        $this->calculateTotalAmount();
    }

    public function getTotalAmount(): int
    {
        return $this->totalAmount;
    }

    #[Groups('admin_list')]
    public function getTotalAmountExcel(): Currency
    {
        return new Currency($this->getTotalAmount());
    }

    public function getBillingAddress(): ?string
    {
        return $this->billingAddress;
    }

    public function setBillingAddress(string $billingAddress): void
    {
        $this->billingAddress = $billingAddress;
    }

    public function getDebtor(): ?Debtor
    {
        return $this->debtor;
    }

    public function setDebtor(?Debtor $debtor): void
    {
        $this->debtor = $debtor;
    }

    public function getBillingEmail(): ?string
    {
        return $this->billingEmail;
    }

    public function setBillingEmail(?string $billingEmail): void
    {
        $this->billingEmail = $billingEmail;
    }

    public function getReceiptNumber(): ?string
    {
        return $this->receiptNumber;
    }

    public function setReceiptNumber(string $receiptNumber): void
    {
        if (null !== $this->receiptNumber) {
            throw new \BadMethodCallException('Receipt number is already set.');
        }

        $this->receiptNumber = $receiptNumber;
    }

    public function getStatus(): string
    {
        return $this->status;
    }

    public function setStatus(string $status): void
    {
        $this->status = $status;
    }

    public function isPaid(): bool
    {
        return self::STATUS_PAID === $this->status;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function setUser(?User $user)
    {
        $this->user = $user;
    }

    public function getPmPaymentTransactionStatus(): ?PmPaymentTransactionStatus
    {
        return $this->pmPaymentTransactionStatus;
    }

    #[Groups('admin_list')]
    public function getUserEmail(): ?string
    {
        return $this->user?->getEmail();
    }

    public function calculateTotalAmount(): void
    {
        $this->totalAmount = array_sum(array_map(fn (PaymentItem $item) => $item->getAmount(), $this->items->toArray()));
    }

    public function setPmPaymentTransactionStatus(?PmPaymentTransactionStatus $pmPaymentTransactionStatus): void
    {
        if ($pmPaymentTransactionStatus && $pmPaymentTransactionStatus->getTxid() !== $this->getPmPaymentTransactionId()) {
            throw new \InvalidArgumentException('Transaction ID does not match.');
        }

        $this->pmPaymentTransactionStatus = $pmPaymentTransactionStatus;
    }

    public function getPmPaymentTransactionId(): ?string
    {
        return $this->pmPaymentTransactionId;
    }

    public function setPmPaymentTransactionId(?string $pmPaymentTransactionId): void
    {
        $this->pmPaymentTransactionId = $pmPaymentTransactionId;
    }

    public function isFinalized(): bool
    {
        return null !== $this->receiptNumber;
    }

    public function createBookEntry(?int $amount = null, ?string $postingRecord = null, ?string $type = null): void
    {
        $type ??= null === $amount ? 'payment' : 'manual';
        $amount ??= $this->totalAmount;
        $postingRecord ??= $this->receiptNumber;

        $this->bookEntries->add(
            new BookEntry($amount, $this->debtor, $this, $type, $postingRecord)
        );
    }

    private function populateAddressFromUser(?User $user): void
    {
        if (!$user instanceof User) {
            return;
        }

        $this->setBillingEmail($user->getEmail());

        if ($postalAddress = $user->getPostalAddresses()->first()) {
            $this->setBillingAddress(<<<EOF
                {$user->getName()}
                {$postalAddress->getStreet()}
                {$postalAddress->getPostalCode()} {$postalAddress->getCity()}
                {$postalAddress->getCountry()}
                EOF);
        }
    }
}
