<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Symfony\Component\Workflow\Transition;

#[ORM\Entity]
#[ORM\Index(columns: ['offer_id', 'createdAt'])]
class OfferLog
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\Column(type: 'string', length: 32, nullable: true)]
    private ?string $transitionName = null;
    #[ORM\Column(type: 'string', length: 32, nullable: true)]
    private ?string $transitionFrom = null;
    #[ORM\Column(type: 'string', length: 32, nullable: true)]
    private ?string $transitionTo = null;

    public function __construct(
        #[ORM\ManyToOne(targetEntity: OfferInterface::class, inversedBy: 'activity')]
        #[ORM\JoinColumn(name: 'offer_id', referencedColumnName: 'id')]
        private OfferInterface $offer,
        #[ORM\ManyToOne(targetEntity: User::class)]
        #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
        private User $createdBy,
        #[ORM\Column(type: 'string', nullable: true)]
        private ?string $comment = null,
        ?Transition $transition = null
    ) {
        if ($transition instanceof Transition) {
            $this->transitionName = $transition->getName();
            $this->transitionFrom = $transition->getFroms()[0] ?? null;
            $this->transitionTo = $transition->getTos()[0] ?? null;
        }

        $this->createdAt = new \DateTimeImmutable();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function getOffer(): OfferInterface
    {
        return $this->offer;
    }

    public function isComment(): bool
    {
        return null !== $this->comment;
    }

    public function isTransition(): bool
    {
        return null !== $this->transitionName;
    }

    public function getComment(): ?string
    {
        return $this->comment;
    }

    public function getTransition(): ?Transition
    {
        if (!$this->isTransition()) {
            return null;
        }

        return new Transition($this->transitionName, $this->transitionFrom ?? [], $this->transitionTo ?? []);
    }

    public function getCreatedBy(): User
    {
        return $this->createdBy;
    }
}
