<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Symfony\Component\Serializer\Annotation\Context;
use Symfony\Component\Serializer\Attribute\Groups;
use Symfony\Component\Serializer\Normalizer\DateTimeNormalizer;

#[ORM\Entity]
#[ORM\Index(columns: ['offer_id', 'end'])]
#[ORM\Index(columns: ['offer_id', 'begin'])]
class OfferDate
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: 'datetime', nullable: true)]
    #[Groups('api:offer:anon')]
    #[Context(normalizationContext: [DateTimeNormalizer::TIMEZONE_KEY => 'Europe/Berlin'])]
    private ?\DateTime $begin = null;

    #[ORM\Column(type: 'datetime', nullable: true)]
    #[Groups('api:offer:anon')]
    #[Context(normalizationContext: [DateTimeNormalizer::TIMEZONE_KEY => 'Europe/Berlin'])]
    private ?\DateTime $end = null;

    public function __construct(
        #[ORM\ManyToOne(targetEntity: OfferInterface::class, inversedBy: 'dates')]
        #[ORM\JoinColumn(name: 'offer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
        private OfferInterface $offer
    ) {
    }

    public function getOffer(): OfferInterface
    {
        return $this->offer;
    }

    public function getBegin(): ?\DateTime
    {
        return $this->begin;
    }

    public function getEnd(): ?\DateTime
    {
        return $this->end;
    }

    public function setBegin(?\DateTime $begin): void
    {
        $this->begin = $begin;
    }

    public function setEnd(?\DateTime $end): void
    {
        $this->end = $end;
    }

    public function toHumanReadable(): string
    {
        if (!$this->begin instanceof \DateTime || !$this->end instanceof \DateTime) {
            return '';
        }

        $begin = $this->begin->setTimezone(new \DateTimeZone('Europe/Berlin'));
        $end = $this->end->setTimezone(new \DateTimeZone('Europe/Berlin'));

        if ($begin->format('Y-m-d') === $end->format('Y-m-d')) {
            return \sprintf('%s - %s', $begin->format('d.m.Y H:i'), $end->format('H:i'));
        }

        return \sprintf('%s - %s', $begin->format('d.m. H:i'), $end->format('d.m.Y H:i'));
    }
}
