<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;

#[ORM\Entity]
#[ORM\UniqueConstraint(columns: ['offer_id', 'code'])]
class FriendCode
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\JoinTable(name: 'FriendCodeToParticipant')]
    #[ORM\JoinColumn(name: 'code_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
    #[ORM\InverseJoinColumn(name: 'participant_id', onDelete: 'CASCADE')]
    #[ORM\ManyToMany(targetEntity: ParticipantInterface::class, inversedBy: 'friendCodes')]
    private Collection $participants;

    public function __construct(
        #[ORM\ManyToOne(targetEntity: OfferInterface::class, inversedBy: 'friendCodes')]
        #[ORM\JoinColumn(name: 'offer_id', referencedColumnName: 'id')]
        private OfferInterface $offer,
        #[ORM\Column(type: 'string')]
        private string $code
    ) {
        $this->createdAt = new \DateTimeImmutable();
        $this->participants = new ArrayCollection();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function getOffer(): OfferInterface
    {
        return $this->offer;
    }

    public function setCode(?string $code): void
    {
        $this->code = (string) $code;
    }

    public function getCode(): string
    {
        return $this->code;
    }

    /**
     * @return Collection<ParticipantInterface>
     */
    public function getParticipants(): Collection
    {
        return $this->participants;
    }

    public function addParticipant(ParticipantInterface $participant): void
    {
        $this->participants[] = $participant;
    }

    public function removeParticipant(ParticipantInterface $participant): void
    {
        $this->participants->removeElement($participant);
    }
}
