<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Facade\PaymentsFacade;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

#[ORM\Entity]
class Debtor implements \Stringable
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: UuidType::NAME)]
    private Uuid $uuid;

    #[ORM\OneToMany(mappedBy: 'debtor', targetEntity: Payment::class)]
    private Collection $payments;

    #[ORM\OneToMany(mappedBy: 'debtor', targetEntity: BookEntry::class)]
    #[ORM\OrderBy(['bookingDate' => 'DESC'])]
    private Collection $bookEntries;

    public function __construct(#[ORM\OneToOne(inversedBy: 'debtor', targetEntity: User::class)]
        #[ORM\JoinColumn(nullable: true, onDelete: 'SET NULL')]
        private ?User $user = null)
    {
        $this->uuid = Uuid::v4();
        $this->payments = new ArrayCollection();
        $this->bookEntries = new ArrayCollection();
    }

    /*
     * For array_unique function
     */
    public function __toString(): string
    {
        return $this->uuid->toRfc4122();
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getUuid(): Uuid
    {
        return $this->uuid;
    }

    public function getUser(): ?User
    {
        return $this->user;
    }

    public function getPayments(): Collection
    {
        return $this->payments;
    }

    public function getBookEntries(): Collection
    {
        return $this->bookEntries;
    }

    public function getBalance(): int
    {
        return array_sum(
            $this->getBookEntries()
                ->filter(fn (BookEntry $e) => !$e->isPayment())
                ->map(fn (BookEntry $e) => $e->getValue())
                ->toArray()
        );
    }

    public function getSummedPayments(): int
    {
        return array_sum(
            $this->getBookEntries()
                ->filter(fn (BookEntry $e) => $e->isPayment())
                ->map(fn (BookEntry $e) => $e->getValue())
                ->toArray()
        );
    }

    /**
     * @return Attendance[]
     */
    public function getAttendancesDue(): array
    {
        if (!$this->user instanceof User) {
            return [];
        }

        $criteria = Criteria::create();
        PaymentsFacade::addCriteria($criteria);

        return array_merge(
            ...$this->user->getParticipants()
            ->map(fn (ParticipantInterface $participant) => $participant->getAttendances()->matching($criteria)->toArray())
            ->toArray()
        );
    }

    public function getDueSum(?EventDispatcherInterface $dispatcher = null): int
    {
        return array_sum(
            array_map(
                fn (Attendance $a) => $a->getOffer()->getFeePayable($a->getParticipant(), $dispatcher),
                $this->getAttendancesDue()
            )
        );
    }
}
