<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Uid\Uuid;

#[ORM\Entity]
class DbafsMedia
{
    #[ORM\Column(type: 'integer', unique: true, nullable: true, options: ['unsigned' => true], columnDefinition: 'INT AUTO_INCREMENT UNIQUE')]
    private ?int $id = null;

    #[ORM\Id]
    #[ORM\Column(type: UuidType::NAME)]
    #[ORM\GeneratedValue(strategy: 'CUSTOM')]
    #[ORM\CustomIdGenerator(class: 'doctrine.uuid_generator')]
    private ?Uuid $uuid = null;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    #[Groups('notification')]
    private \DateTimeInterface $createdAt;

    #[ORM\Column(type: UuidType::NAME, nullable: true)]
    private ?Uuid $pid = null;

    #[ORM\Column(type: 'string')]
    private ?string $path = null;
    #[ORM\Column(type: 'string', length: 16)]
    private ?string $type = null;
    #[ORM\Column(type: 'string', length: 64)]
    private ?string $hash = null;

    #[ORM\Column(type: 'integer', length: 13, nullable: true)]
    private ?int $lastModified = null;

    #[ORM\ManyToOne(targetEntity: User::class)]
    #[ORM\JoinColumn(name: 'uploaded_by', referencedColumnName: 'id', onDelete: 'CASCADE')]
    private ?User $uploadedBy = null;

    #[ORM\OneToMany(targetEntity: OfferMedia::class, mappedBy: 'media', cascade: ['remove'])]
    private Collection $offers;

    #[ORM\Column(type: 'float', options: ['unsigned' => true, 'default' => 0])]
    private float $importantPartX = 0;
    #[ORM\Column(type: 'float', options: ['unsigned' => true, 'default' => 0])]
    private float $importantPartY = 0;
    #[ORM\Column(type: 'float', options: ['unsigned' => true, 'default' => 0])]
    private float $importantPartWidth = 0;
    #[ORM\Column(type: 'float', options: ['unsigned' => true, 'default' => 0])]
    private float $importantPartHeight = 0;

    #[ORM\Column(type: 'string', nullable: true)]
    private ?string $alt = null;
    #[ORM\Column(type: 'string', nullable: true)]
    private ?string $creator = null;

    public function __construct()
    {
        $this->createdAt = new \DateTimeImmutable();
        $this->offers = new ArrayCollection();
    }

    public function getUuid(): ?Uuid
    {
        return $this->uuid;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function setCreatedAt(\DateTimeInterface $createdAt): void
    {
        $this->createdAt = $createdAt;
    }

    public function getOffers(): Collection
    {
        return $this->offers;
    }

    public function getUploadedBy(): ?User
    {
        return $this->uploadedBy;
    }

    public function setUploadedBy(?User $uploadedBy): void
    {
        $this->uploadedBy = $uploadedBy;
    }

    public function getLastModified(): \DateTimeInterface
    {
        return new \DateTimeImmutable('@'.$this->lastModified);
    }

    public function setLastModified(?int $lastModified): void
    {
        $this->lastModified = $lastModified;
    }

    public function getImportantPart(): array
    {
        return [
            'x' => $this->importantPartX,
            'y' => $this->importantPartY,
            'width' => $this->importantPartWidth,
            'height' => $this->importantPartHeight,
        ];
    }

    public function setImportantPart(array $importantPart)
    {
        $this->importantPartX = $importantPart['x'] ?? 0;
        $this->importantPartY = $importantPart['y'] ?? 0;
        $this->importantPartWidth = $importantPart['width'] ?? 0;
        $this->importantPartHeight = $importantPart['height'] ?? 0;
    }

    public function getAlt(): ?string
    {
        return $this->alt;
    }

    public function setAlt(?string $alt): void
    {
        $this->alt = $alt;
    }

    public function getCreator(): ?string
    {
        return $this->creator;
    }

    public function setCreator(?string $creator): void
    {
        $this->creator = $creator;
    }
}
