<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Symfony\Bridge\Doctrine\IdGenerator\UuidGenerator;
use Symfony\Bridge\Doctrine\Types\UuidType;
use Symfony\Component\Uid\Uuid;

#[ORM\Entity]
class ConsentForm
{
    #[ORM\Id]
    #[ORM\GeneratedValue(strategy: 'CUSTOM')]
    #[ORM\Column(type: UuidType::NAME, unique: true)]
    #[ORM\CustomIdGenerator(class: UuidGenerator::class)]
    private Uuid $id;

    #[ORM\ManyToOne(targetEntity: OfferInterface::class, inversedBy: 'consentForms')]
    #[ORM\JoinColumn(name: 'offer_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
    private ?OfferInterface $offer = null;

    #[ORM\ManyToOne(targetEntity: Edition::class, inversedBy: 'consentForms')]
    #[ORM\JoinColumn(name: 'edition_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
    private ?Edition $edition = null;

    #[ORM\OneToOne(targetEntity: ConsentText::class, cascade: ['persist'])]
    #[ORM\JoinColumn(name: 'consent_text', referencedColumnName: 'id', onDelete: 'SET NULL')]
    private ?ConsentText $consentText = null;

    #[ORM\Column(type: 'string', length: 12, options: ['default' => 'participant'])]
    private string $subject = 'participant';

    #[ORM\Column(type: 'string', length: 16, options: ['default' => 'compulsory'])]
    private string $compulsory = 'compulsory';

    #[ORM\OneToMany(mappedBy: 'consentForm', targetEntity: Consent::class)]
    #[ORM\JoinColumn(referencedColumnName: 'id', onDelete: 'SET NULL')]
    private Collection $signatures;

    public function __construct(?Uuid $uuid = null)
    {
        if ($uuid instanceof Uuid) {
            $this->id = $uuid;
        }

        $this->signatures = new ArrayCollection();
    }

    public function getId(): Uuid
    {
        return $this->id;
    }

    public function getOffer(): ?OfferInterface
    {
        return $this->offer;
    }

    public function setOffer(?OfferInterface $offer): void
    {
        $this->offer = $offer;
    }

    public function getEdition(): ?Edition
    {
        return $this->edition;
    }

    public function setEdition(?Edition $edition): void
    {
        $this->edition = $edition;
    }

    public function getConsentText(): ?ConsentText
    {
        return $this->consentText;
    }

    public function setConsentText(?ConsentText $consentText): void
    {
        $this->consentText = $consentText;
    }

    public function getText(): ?string
    {
        return $this->consentText?->getText();
    }

    public function setText(?string $text): void
    {
        if (null === $text) {
            $this->consentText = null;

            return;
        }

        if ($this->consentText?->getText() === $text) {
            return;
        }

        $this->consentText = new ConsentText($text);
    }

    public function getSubject(): string
    {
        return $this->subject;
    }

    public function setSubject(string $subject): void
    {
        $this->subject = $subject;
    }

    public function getCompulsory(): string
    {
        return $this->compulsory;
    }

    public function setCompulsory(?string $compulsory): void
    {
        if (null === $compulsory) {
            return;
        }

        $this->compulsory = $compulsory;
    }

    public function getSignatures(): Collection
    {
        return $this->signatures;
    }
}
