<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;

#[ORM\Entity]
#[ORM\InheritanceType('SINGLE_TABLE')]
#[ORM\DiscriminatorColumn(name: 'type', type: 'string')]
#[ORM\DiscriminatorMap(['form_consent' => Consent::class, 'edition_agreement_letter' => AgreementLetterSignature::class, 'host_consent' => HostConsent::class, 'photo_permission' => PhotoPermission::class, 'medical_care_objection' => MedicalCareObjection::class])]
class Consent
{
    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;

    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\Column(type: 'datetime_immutable', nullable: true)]
    private ?\DateTimeInterface $signedAt = null;

    #[ORM\Column(type: 'datetime_immutable', nullable: true)]
    private ?\DateTimeInterface $revokedAt = null;

    public function __construct(
        #[ORM\ManyToOne(targetEntity: User::class, inversedBy: 'consents')]
        #[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'cascade')]
        private User $user,
        #[ORM\Column(type: 'string', length: 64, nullable: true)]
        private ?string $hash = null,
        #[ORM\Column(type: 'datetime', nullable: true)]
        private ?\DateTimeInterface $validUntil = null,
        #[ORM\ManyToOne(targetEntity: ParticipantInterface::class, inversedBy: 'consents')]
        #[ORM\JoinColumn(name: 'participant_id', referencedColumnName: 'id', onDelete: 'cascade')]
        private ?ParticipantInterface $participant = null,
        #[ORM\ManyToOne(targetEntity: ConsentForm::class, inversedBy: 'signatures')]
        #[ORM\JoinColumn(name: 'consent_form', referencedColumnName: 'id', onDelete: 'SET NULL')]
        private ?ConsentForm $consentForm = null,
        #[ORM\ManyToOne(targetEntity: ConsentText::class, cascade: ['persist'])]
        #[ORM\JoinColumn(name: 'consent_text', referencedColumnName: 'id', onDelete: 'SET NULL')]
        private ?ConsentText $consentText = null,
        #[ORM\Column(type: 'integer', options: ['default' => 1])]
        private int $decision = 1,
    ) {
        $this->createdAt = new \DateTimeImmutable();
        if (1 === $this->decision) {
            $this->signedAt = new \DateTimeImmutable();
        }
        if (null === $this->consentText && null !== $this->consentForm) {
            $this->consentText = $this->consentForm->getConsentText();
        }
        $this->calculateHash();
    }

    public function calculateHash()
    {
        if (null === $this->consentText) {
            return;
        }

        $this->hash = hash('sha256', $this->consentText->getText());
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function getSignedAt(): ?\DateTimeInterface
    {
        return $this->signedAt;
    }

    public function getDecision(): int
    {
        return $this->decision;
    }

    public function setDecision(int $decision): void
    {
        $this->decision = $decision;

        if (1 === $decision && null === $this->signedAt) {
            $this->signedAt = new \DateTimeImmutable();
        }

        if (1 === $decision) {
            $this->unrevoke();
        }
    }

    public function getUser(): User
    {
        return $this->user;
    }

    public function getHash(): ?string
    {
        return $this->consentText?->getHash() ?? $this->hash;
    }

    public function getEdition(): ?Edition
    {
        if (null === $this->consentForm) {
            return null;
        }

        if (null === $this->consentForm->getEdition()) {
            return null;
        }

        return $this->consentForm->getEdition();
    }

    public function getOffer(): ?OfferInterface
    {
        if (null === $this->consentForm) {
            return null;
        }

        if (null === $this->consentForm->getOffer()) {
            return null;
        }

        return $this->consentForm->getOffer();
    }

    public function getConsentForm(): ?ConsentForm
    {
        return $this->consentForm;
    }

    public function getConsentText(): ?ConsentText
    {
        return $this->consentText;
    }

    public function isValid(?string $hash = null): bool
    {
        if ($this->validUntil instanceof \DateTimeInterface && $this->validUntil < new \DateTimeImmutable()) {
            return false;
        }

        if (null !== $this->consentText) {
            return null === $hash || hash_equals($this->consentText->getHash(), (string) $hash);
        }

        return null === $this->hash || hash_equals($this->hash, (string) $hash);
    }

    public function getRevokedAt(): ?\DateTimeInterface
    {
        return $this->revokedAt;
    }

    public function getValidUntil(): ?\DateTimeInterface
    {
        return $this->validUntil;
    }

    public function isRevoked(): bool
    {
        return $this->revokedAt instanceof \DateTimeInterface;
    }

    public function revoke(): void
    {
        if (!$this->getRevokedAt() instanceof \DateTimeInterface) {
            $this->revokedAt = new \DateTimeImmutable();
        }
    }

    public function unrevoke(): void
    {
        $this->revokedAt = null;
    }

    public function getParticipant(): ?ParticipantInterface
    {
        return $this->participant;
    }

    public function getType(): string
    {
        if ($this instanceof AgreementLetterSignature) {
            return 'edition_agreement_letter';
        }

        if ($this instanceof HostConsent) {
            return 'host_consent';
        }

        if ($this instanceof PhotoPermission) {
            return 'photo_permission';
        }

        if ($this instanceof MedicalCareObjection) {
            return 'medical_care_objection';
        }

        if (null !== $this->consentForm) {
            return 'form_consent';
        }

        return 'consent';
    }
}
