<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

#[ORM\Entity]
class ApiToken
{
    public const string PERSONAL_ACCESS_TOKEN_PREFIX = 'pat-';

    #[ORM\Id]
    #[ORM\GeneratedValue]
    #[ORM\Column(type: 'integer', options: ['unsigned' => true])]
    private ?int $id = null;
    #[ORM\Column(type: 'datetime_immutable', options: ['default' => 'CURRENT_TIMESTAMP'])]
    private \DateTimeInterface $createdAt;

    #[ORM\Column(type: 'datetime', nullable: true)]
    private ?\DateTimeInterface $validUntil = null;

    public function __construct(#[ORM\Column(type: 'string', length: 255, nullable: false)]
        private string $locator, #[\SensitiveParameter] #[ORM\Column(type: 'string', length: 255, nullable: false)] private string $secret, #[ORM\ManyToOne(targetEntity: User::class, inversedBy: 'apiTokens')]
        #[ORM\JoinColumn(name: 'owner_id', referencedColumnName: 'id', onDelete: 'cascade')]
        private User $ownedBy, #[ORM\Column(type: 'json')]
        private array $scopes)
    {
        $this->createdAt = new \DateTimeImmutable();
    }

    public static function fromToken(#[\SensitiveParameter] $token, array $scopes, User $owner, UserPasswordHasherInterface $hasher): self
    {
        if (!str_starts_with((string) $token, self::PERSONAL_ACCESS_TOKEN_PREFIX)) {
            throw new \InvalidArgumentException('$token must start with '.self::PERSONAL_ACCESS_TOKEN_PREFIX);
        }

        if (1 !== substr_count((string) $token, '.')) {
            throw new \InvalidArgumentException('$token must contain exactly one `.` character, delimiting the locator from the secret value');
        }

        [$locator, $secret] = explode('.', (string) $token, 2);
        $secret = $hasher->hashPassword($owner, $secret);

        return new self($locator, $secret, $owner, $scopes);
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function getOwnedBy(): User
    {
        return $this->ownedBy;
    }

    public function getLocator(): string
    {
        return $this->locator;
    }

    public function getSecret(): string
    {
        return $this->secret;
    }

    public function getScopes(): array
    {
        return $this->scopes;
    }

    public function getValidUntil(): ?\DateTimeInterface
    {
        return $this->validUntil;
    }

    public function getCreatedAt(): \DateTimeInterface
    {
        return $this->createdAt;
    }

    public function isValid(): bool
    {
        return !($this->validUntil instanceof \DateTimeInterface && $this->validUntil < new \DateTimeImmutable());
    }
}
