<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\DependencyInjection;

use Contao\CoreBundle\DependencyInjection\Filesystem\ConfigureFilesystemInterface;
use Contao\CoreBundle\DependencyInjection\Filesystem\FilesystemConfiguration;
use Doctrine\Bundle\DoctrineBundle\Attribute\AsEntityListener;
use Ferienpass\CoreBundle\Attribute\AsOfferEntityListener;
use Ferienpass\CoreBundle\Attribute\AsParticipantEntityListener;
use Ferienpass\CoreBundle\Cron\SendRemindersListener;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Export\Offer\OffersExportInterface;
use Ferienpass\CoreBundle\Export\Offer\PrintSheet\PdfExport;
use Ferienpass\CoreBundle\Export\Offer\Xml\XmlExport;
use Ferienpass\CoreBundle\Export\ParticipantList\WordExport;
use Ferienpass\CoreBundle\Messenger\MessageLogMiddleware;
use Ferienpass\CoreBundle\Monolog\MessengerLogHandler;
use Ferienpass\CoreBundle\Monolog\NotificationLogHandler;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Ferienpass\CoreBundle\Repository\ResetPasswordRequestRepository;
use Ferienpass\CoreBundle\Webhook\PmPaymentRequestParser;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ChildDefinition;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Extension\Extension;
use Symfony\Component\DependencyInjection\Extension\PrependExtensionInterface;
use Symfony\Component\DependencyInjection\Loader\PhpFileLoader;
use Symfony\Component\ExpressionLanguage\ExpressionLanguage;

final class FerienpassCoreExtension extends Extension implements PrependExtensionInterface, ConfigureFilesystemInterface
{
    use PersistenceExtensionTrait;

    public function load(array $configs, ContainerBuilder $container): void
    {
        $config = $this->processConfiguration(new Configuration(), $configs);

        $loader = new PhpFileLoader($container, new FileLocator(__DIR__.'/../../config'));
        $loader->load('services.php');

        $this->configurePersistence($config['entities'], $container);
        $container->addAliases([
            OfferRepositoryInterface::class => 'ferienpass.repository.offer',
            ParticipantRepositoryInterface::class => 'ferienpass.repository.participant',
        ]);

        // Parameters
        $container->setParameter('ferienpass.logos_dir', $config['logos_dir']);
        $container->setParameter('ferienpass.images_dir', $config['images_dir']);
        $container->setParameter('ferienpass.friends_booking', $config['friends_booking']);
        $container->setParameter('ferienpass.reply_address', $config['reply_address']);
        $container->setParameter('ferienpass.preferred_postal_codes', $config['preferred_postal_codes']);
        $container->setParameter('ferienpass.require_postal_address', $config['require_postal_address']);
        $container->setParameter('ferienpass.erase_data_period', $config['erase_data_after']);
        $container->setParameter('ferienpass.attendance_remind_period', $config['remind_attendance_before']);
        $container->setParameter('ferienpass.payment_allow_select', $config['payment_allow_select']);
        $container->setParameter('ferienpass.age_tolerance', $config['age_tolerance']);
        $container->setParameter('ferienpass.withdraw_grace_period', $config['withdraw_grace_period']);
        $container->setParameter('ferienpass.no_withdraw_after_grace_period', $config['no_withdraw_after_grace_period']);
        $container->setParameter('ferienpass.no_withdraw_after_application_deadline', $config['no_withdraw_after_application_deadline']);
        $container->setParameter('ferienpass.participant_list_sort', $config['participant_list_sort']);

        $expressionLanguage = new ExpressionLanguage();
        $container->setParameter('ferienpass.receipt_number_prefix', null === $config['receipt_number_prefix'] ? '' : $expressionLanguage->evaluate($config['receipt_number_prefix'], ['date' => new \DateTimeImmutable()]));

        // Feature flags
        if (false === $config['remind_attendance']) {
            $container->removeDefinition(SendRemindersListener::class);
        }

        // Injection
        if (isset($config['export'])) {
            foreach ($config['export']['pdf'] as $name => $pdfConfig) {
                $definition = new Definition(PdfExport::class)
                    ->setAutowired(true)
                    ->setArguments([
                        $pdfConfig['mpdf_config'] ?? [],
                        $pdfConfig['template'] ?? '',
                    ])
                ;
                $definition->addTag(OffersExportInterface::class, ['name' => $name.'.pdf']);
                $container->setDefinition('ferienpass.exporter.pdf.'.$name, $definition);
            }

            foreach ($config['export']['xml'] as $name => $template) {
                $definition = new Definition(XmlExport::class)->setAutowired(true)->setArguments([$template]);
                $definition->addTag(OffersExportInterface::class, ['name' => $name.'.xml']);
                $container->setDefinition('ferienpass.exporter.xml.'.$name, $definition);
            }
        }

        $docxParticipantList = $container->getDefinition(WordExport::class);
        $docxParticipantList->setArgument(2, $config['export']['participant_list_docx'] ?? null);

        $container->registerAttributeForAutoconfiguration(AsOfferEntityListener::class, static function (ChildDefinition $definition, AsEntityListener $attribute) use ($container) {
            $definition->addTag('doctrine.orm.entity_listener', [
                'event' => $attribute->event,
                'method' => $attribute->method,
                'lazy' => $attribute->lazy,
                'entity_manager' => $attribute->entityManager,
                'entity' => $container->getParameter('ferienpass.model.offer.class'),
                'priority' => $attribute->priority,
            ]);
        });
        $container->registerAttributeForAutoconfiguration(AsParticipantEntityListener::class, static function (ChildDefinition $definition, AsEntityListener $attribute) use ($container) {
            $definition->addTag('doctrine.orm.entity_listener', [
                'event' => $attribute->event,
                'method' => $attribute->method,
                'lazy' => $attribute->lazy,
                'entity_manager' => $attribute->entityManager,
                'entity' => $container->getParameter('ferienpass.model.participant.class'),
                'priority' => $attribute->priority,
            ]);
        });
    }

    public function getAlias(): string
    {
        return 'ferienpass';
    }

    public function prepend(ContainerBuilder $container): void
    {
        $this->prependTwigBundle($container);

        $container->prependExtensionConfig('monolog', [
            'channels' => ['ferienpass_messages', 'ferienpass_notifications'],
            'handlers' => [
                'messenger' => [
                    'type' => 'service',
                    'id' => MessengerLogHandler::class,
                    'channels' => ['ferienpass_messages'],
                ],
                'notifications' => [
                    'type' => 'service',
                    'id' => NotificationLogHandler::class,
                    'channels' => ['ferienpass_notifications'],
                ],
            ],
        ]);

        $container->prependExtensionConfig('framework', [
            'mailer' => [
                'envelope' => [
                    'sender' => '%env(ADMIN_EMAIL)%',
                ],
            ],
            'webhook' => [
                'routing' => [
                    'pmPayment' => [
                        'service' => PmPaymentRequestParser::class,
                    ],
                ],
            ],
            'http_client' => [
                'scoped_clients' => [
                    'pmPayment.client' => [
                        'base_uri' => 'https://www.payment.govconnect.de',
                    ],
                    'brevo.client' => [
                        'base_uri' => 'https://api.brevo.com/v3',
                        'headers' => [
                            'accept' => 'application/json',
                            'api-key' => '%env(BREVO_API_KEY)%',
                        ],
                    ],
                ],
            ],
            'rate_limiter' => [
                'friend_code' => [
                    'policy' => 'sliding_window',
                    'limit' => 10,
                    'interval' => '1 day',
                ],
            ],
        ]);

        // TODO only register if NOT customized
        $container->prependExtensionConfig('doctrine', [
            'orm' => [
                'entity_managers' => [
                    'default' => [
                        'schema_ignore_classes' => [
                            Offer::class,
                            Participant::class,
                        ],
                    ],
                ],
            ],
        ]);

        $container->prependExtensionConfig('framework', [
            'messenger' => [
                'default_bus' => 'command.bus',
                'buses' => [
                    'command.bus' => [
                        'middleware' => [
                            MessageLogMiddleware::class,
                            'doctrine_close_connection',
                            'doctrine_transaction',
                            'router_context',
                        ],
                    ],
                    'event.bus' => [
                        'default_middleware' => [
                            'enabled' => true,
                            'allow_no_handlers' => true,
                        ],
                        'middleware' => [
                            MessageLogMiddleware::class,
                            'doctrine_close_connection',
                            'router_context',
                        ],
                    ],
                ],
            ],
        ]);

        $container->prependExtensionConfig('symfonycasts_reset_password', [
            'request_password_repository' => ResetPasswordRequestRepository::class,
        ]);

        $this->prependWorkflow($container);
    }

    public function configureFilesystem(FilesystemConfiguration $config): void
    {
        $config
            ->mountLocalAdapter('storage/media', 'storage/media', 'media')
            ->addVirtualFilesystem('media', 'storage/media')
        ;

        $config->addDefaultDbafs('media', 'DbafsMedia');

        $config
            ->mountLocalAdapter('storage/attachments', 'storage/attachments', 'attachments')
            ->addVirtualFilesystem('attachments', 'storage/attachments')
        ;

        $config->addDefaultDbafs('attachments', 'DbafsAttachment');

        $config
            ->mountLocalAdapter('storage/logos', 'storage/logos', 'logos')
            ->addVirtualFilesystem('logos', 'storage/logos')
        ;

        $config->addDefaultDbafs('logos', 'DbafsLogo');

        $config
            ->mountLocalAdapter('storage/export', 'storage/export', 'export')
            ->addVirtualFilesystem('export', 'storage/export')
        ;
    }

    private function prependTwigBundle(ContainerBuilder $container): void
    {
        $container->prependExtensionConfig('twig', [
            'paths' => [
                __DIR__.'/../../templates/email' => 'email',
                __DIR__.'/../../email/css' => 'email_styles',
            ],
            'form_themes' => [
                '@FerienpassCore/form/custom_types.html.twig',
                '@FerienpassCore/form/theme.html.twig',
            ],
        ]);
    }

    private function prependWorkflow(ContainerBuilder $container): void
    {
        $container->prependExtensionConfig('framework', config: [
            'workflows' => [
                'offer' => [
                    'type' => 'workflow',
                    'marking_store' => ['type' => 'method', 'property' => 'status'],
                    'supports' => [OfferInterface::class],
                    'initial_marking' => OfferInterface::STATE_DRAFT,
                    'places' => [OfferInterface::STATE_COMPLETED, OfferInterface::STATE_DRAFT, OfferInterface::STATE_FINALIZED, OfferInterface::STATE_REVIEWED, OfferInterface::STATE_PUBLISHED, OfferInterface::STATE_CANCELLED],
                    'transitions' => [
                        OfferInterface::TRANSITION_COMPLETE => [
                            'guard' => "is_granted('complete', subject)",
                            'to' => OfferInterface::STATE_COMPLETED,
                        ],
                        OfferInterface::TRANSITION_FINALIZE => [
                            'guard' => "is_granted('finalize', subject)",
                            'from' => OfferInterface::STATE_DRAFT,
                            'to' => OfferInterface::STATE_FINALIZED,
                        ],
                        OfferInterface::TRANSITION_TO_DRAFT => [
                            'guard' => "is_granted('to_draft', subject)",
                            'from' => OfferInterface::STATE_FINALIZED,
                            'to' => OfferInterface::STATE_DRAFT,
                        ],
                        OfferInterface::TRANSITION_TO_DRAFT.'_2' => [
                            'guard' => "is_granted('to_draft', subject)",
                            'from' => OfferInterface::STATE_REVIEWED,
                            'to' => OfferInterface::STATE_DRAFT,
                        ],
                        OfferInterface::TRANSITION_APPROVE => [
                            'guard' => "is_granted('approve', subject)",
                            'from' => OfferInterface::STATE_FINALIZED,
                            'to' => OfferInterface::STATE_REVIEWED,
                        ],
                        OfferInterface::TRANSITION_UNAPPROVE => [
                            'guard' => "is_granted('unapprove', subject)",
                            'from' => OfferInterface::STATE_REVIEWED,
                            'to' => OfferInterface::STATE_FINALIZED,
                        ],
                        OfferInterface::TRANSITION_PUBLISH => [
                            'guard' => "is_granted('publish', subject)",
                            'from' => OfferInterface::STATE_REVIEWED,
                            'to' => OfferInterface::STATE_PUBLISHED,
                        ],
                        OfferInterface::TRANSITION_CANCEL => [
                            'guard' => "is_granted('cancel', subject)",
                            'to' => [OfferInterface::STATE_CANCELLED],
                        ],
                        OfferInterface::TRANSITION_RELAUNCH => [
                            'guard' => "is_granted('relaunch', subject)",
                            'from' => OfferInterface::STATE_CANCELLED,
                        ],
                        OfferInterface::TRANSITION_UNPUBLISH => [
                            'guard' => "is_granted('unpublish', subject)",
                            'from' => OfferInterface::STATE_PUBLISHED,
                            'to' => OfferInterface::STATE_REVIEWED,
                        ],
                    ],
                ],
                'attendance' => [
                    'type' => 'state_machine',
                    'marking_store' => ['type' => 'method', 'property' => 'status'],
                    'supports' => [Attendance::class],
                    'initial_marking' => Attendance::STATUS_WAITING,
                    'places' => [Attendance::STATUS_WAITING, Attendance::STATUS_CONFIRMED, Attendance::STATUS_WAITLISTED, Attendance::STATUS_REJECTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED, Attendance::STATUS_NO_SHOW, Attendance::STATUS_PARTICIPATED],
                    'transitions' => [
                        Attendance::TRANSITION_CONFIRM => [
                            'from' => [Attendance::STATUS_WAITING, Attendance::STATUS_WAITLISTED, Attendance::STATUS_REJECTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED],
                            'to' => Attendance::STATUS_CONFIRMED,
                        ],
                        Attendance::TRANSITION_WAITLIST => [
                            'from' => [Attendance::STATUS_WAITING, Attendance::STATUS_CONFIRMED, Attendance::STATUS_REJECTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED],
                            'to' => Attendance::STATUS_WAITLISTED,
                        ],
                        Attendance::TRANSITION_WITHDRAW => [
                            'from' => [Attendance::STATUS_WAITING, Attendance::STATUS_CONFIRMED, Attendance::STATUS_REJECTED, Attendance::STATUS_WAITLISTED],
                            'to' => Attendance::STATUS_WITHDRAWN,
                        ],
                        Attendance::TRANSITION_RESET => [
                            'from' => [Attendance::STATUS_CONFIRMED, Attendance::STATUS_REJECTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED, Attendance::STATUS_WAITLISTED],
                            'to' => Attendance::STATUS_WAITING,
                        ],
                        Attendance::TRANSITION_REJECT => [
                            'from' => [Attendance::STATUS_CONFIRMED, Attendance::STATUS_WAITLISTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED, Attendance::STATUS_WAITING],
                            'to' => Attendance::STATUS_REJECTED,
                        ],
                        Attendance::TRANSITION_UNFULFILL => [
                            'from' => [Attendance::STATUS_CONFIRMED, Attendance::STATUS_WAITLISTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_REJECTED, Attendance::STATUS_WAITING],
                            'to' => Attendance::STATUS_UNFULFILLED,
                        ],
                        Attendance::TRANSITION_NO_SHOW => [
                            'from' => [Attendance::STATUS_CONFIRMED],
                            'to' => Attendance::STATUS_NO_SHOW,
                        ],
                        Attendance::TRANSITION_PARTICIPATED => [
                            'from' => [Attendance::STATUS_CONFIRMED, Attendance::STATUS_REJECTED, Attendance::STATUS_WITHDRAWN, Attendance::STATUS_UNFULFILLED, Attendance::STATUS_WAITLISTED],
                            'to' => Attendance::STATUS_PARTICIPATED,
                        ],
                    ],
                ],
                'payment' => [
                    'type' => 'state_machine',
                    'marking_store' => ['type' => 'method', 'property' => 'status'],
                    'supports' => [Payment::class],
                    'initial_marking' => Payment::STATUS_CREATED,
                    'places' => [Payment::STATUS_CREATED, Payment::STATUS_PAID, 'unpaid', Payment::STATUS_FAILED, Payment::STATUS_ABANDONED],
                    'transitions' => [
                        Payment::TRANSITION_PAY => [
                            'from' => [Payment::STATUS_CREATED, 'unpaid', Payment::STATUS_FAILED],
                            'to' => Payment::STATUS_PAID,
                        ],
                        Payment::TRANSITION_FAIL => [
                            'from' => [Payment::STATUS_CREATED, Payment::STATUS_PAID, 'unpaid'],
                            'to' => Payment::STATUS_FAILED,
                        ],
                        Payment::TRANSITION_ABANDON => [
                            'from' => [Payment::STATUS_CREATED, 'unpaid'],
                            'to' => Payment::STATUS_ABANDONED,
                        ],
                    ],
                ],
            ],
        ]);
    }
}
