<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\DependencyInjection;

use Ferienpass\CoreBundle\Entity\Offer;
use Ferienpass\CoreBundle\Entity\Participant;
use Ferienpass\CoreBundle\Repository\OfferRepository;
use Ferienpass\CoreBundle\Repository\ParticipantRepository;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

final class Configuration implements ConfigurationInterface
{
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('ferienpass');
        $treeBuilder->getRootNode()
            ->children()
                ->arrayNode('entities')
                    ->addDefaultsIfNotSet()
                    ->children()
                        ->arrayNode('offer')
                            ->addDefaultsIfNotSet()
                            ->children()
                                ->scalarNode('model')->defaultValue(Offer::class)->end()
                                ->scalarNode('repository')->defaultValue(OfferRepository::class)->end()
                            ->end()
                        ->end()
                        ->arrayNode('participant')
                            ->addDefaultsIfNotSet()
                            ->children()
                                ->scalarNode('model')->defaultValue(Participant::class)->end()
                                ->scalarNode('repository')->defaultValue(ParticipantRepository::class)->end()
                            ->end()
                        ->end()
                    ->end()
                ->end()
                ->scalarNode('logos_dir')
                    ->defaultValue('%kernel.project_dir%/files/logos')
                ->end()
                ->scalarNode('images_dir')
                    ->defaultValue('%kernel.project_dir%/files/bilder')
                ->end()
                ->scalarNode('receipt_number_prefix')
                    ->defaultNull()
                ->end()
                ->scalarNode('reply_address')
                    ->defaultNull()
                ->end()
                ->booleanNode('friends_booking')
                    ->defaultFalse()
                ->end()
                ->arrayNode('preferred_postal_codes')
                    ->scalarPrototype()->end()
                ->end()
                ->scalarNode('erase_data_after')
                    ->info('Participants become listed for delete after "NN weeks/days/months" (defaults to 50 weeks).')
                    ->defaultValue('50 WEEK')
                    ->validate()
                        ->always(function ($v) {
                            if (!preg_match('/\d+ (weeks?|days?|months?)/', $v)) {
                                throw new \InvalidArgumentException(\sprintf('Malformed date interval "%s"', $v));
                            }

                            $v = str_replace(['weeks', 'months', 'days'], ['week', 'month', 'day'], $v);

                            return strtoupper($v);
                        })
                    ->end()
                ->end()
                ->booleanNode('require_postal_address')->defaultFalse()->end()
                ->booleanNode('remind_attendance')->defaultTrue()->end()
                ->scalarNode('remind_attendance_before')
                    ->info('Remind confirmed attendances before XX (defaults to 1.5 days).')
                    ->defaultValue('36 hours')
                    ->validate()
                        ->always(function ($v) {
                            // Input is expected to be relative date string.
                            // User input: 36 hours
                            // Relative date string: +36 hours
                            $v = \sprintf('+%s', $v);
                            try {
                                new \DateTimeImmutable($v);
                            } catch (\DateMalformedStringException) {
                                throw new \InvalidArgumentException(\sprintf('Malformed date modifier string "%s"', $v));
                            }

                            return $v;
                        })
                    ->end()
                ->end()
                ->booleanNode('payment_allow_select')->defaultTrue()->end()
                ->scalarNode('age_tolerance')
                    ->info('Allow participant aged +- this time interval.')
                    ->example('90 days')
                    ->defaultNull()
                    ->validate()
                        ->always(function ($v) {
                            if (null === $v) {
                                return null;
                            }

                            try {
                                new \DateTimeImmutable(\sprintf('+%s', $v));
                            } catch (\DateMalformedStringException) {
                                throw new \InvalidArgumentException(\sprintf('Malformed date modifier string "%s"', $v));
                            }

                            return $v;
                        })
                    ->end()
                ->end()
                ->scalarNode('withdraw_grace_period')
                    ->info('Attendances can be withdrawn until XX before start of the event before they are charged full price.')
                    ->example('3 days')
                    ->defaultNull()
                    ->validate()
                        ->always(function ($v) {
                            if (null === $v) {
                                return null;
                            }
                            // Input is expected to be relative date string.
                            // User input: 36 hours
                            // Relative date string: +36 hours
                            $v = \sprintf('+%s', $v);
                            try {
                                new \DateTimeImmutable($v);
                            } catch (\DateMalformedStringException) {
                                throw new \InvalidArgumentException(\sprintf('Malformed date modifier string "%s"', $v));
                            }

                            return $v;
                        })
                    ->end()
                ->end()
                ->booleanNode('no_withdraw_after_grace_period')->defaultFalse()->end()
                ->booleanNode('no_withdraw_after_application_deadline')->defaultFalse()->end()
                ->enumNode('participant_list_sort')
                    ->values(['firstname', 'lastname'])
                    ->defaultValue('lastname')
                ->end()
            ->append($this->addExportNode())
            ->end()
        ;

        return $treeBuilder;
    }

    private function addExportNode()
    {
        return new TreeBuilder('export')->getRootNode()
            ->children()
                ->scalarNode('participant_list_docx')->end()
                ->arrayNode('pdf')
                    ->arrayPrototype()
                        ->children()
                            ->scalarNode('template')
                            ->end()
                            ->arrayNode('mpdf_config')
                                ->variablePrototype()
                                ->end()
                            ->end()
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('xml')
                    ->useAttributeAsKey('name')
                    ->scalarPrototype()->end()
                ->end()
            ->end();
    }
}
