<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Cron;

use Contao\CoreBundle\Cron\Cron;
use Contao\CoreBundle\DependencyInjection\Attribute\AsCronJob;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Query\Expr\Join;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Message\RemindAttendance;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\MessageBusInterface;

#[AsCronJob('hourly')]
class SendRemindersListener
{
    public function __construct(private readonly AttendanceRepository $attendances, private readonly MessageBusInterface $commandBus, #[Autowire(param: 'ferienpass.attendance_remind_period')] private readonly string $noticePeriod)
    {
    }

    public function __invoke(string $scope): void
    {
        if (Cron::SCOPE_WEB === $scope) {
            return;
        }

        // Notice period: 1.5 days
        $noticePeriod = new \DateTimeImmutable($this->noticePeriod);
        $published = OfferInterface::STATE_PUBLISHED;
        $cancelled = OfferInterface::STATE_CANCELLED;

        $qb = $this->attendances->createQueryBuilder('a');

        // Subquery: Attendances being notified already
        $sub = $this->attendances->createQueryBuilder('a1')
            ->leftJoin('a1.messengerLogs', 'm', Join::WITH, 'm.message = :message')
            ->andWhere('m IS NOT NULL')
            ->addGroupBy('a1.id')
        ;

        $attendances = $qb
            ->innerJoin('a.offer', 'o')
            ->innerJoin('o.dates', 'd')
            ->groupBy('o.id')

            ->andWhere($qb->expr()->notIn('a.id', $sub->getDQL()))
            ->setParameter('message', RemindAttendance::class)

            ->andWhere('a.status = :status')
            ->setParameter('status', Attendance::STATUS_CONFIRMED)

            ->andWhere("JSON_CONTAINS_PATH(o.status, 'one', '$.$published') = 1")
            ->andWhere("JSON_CONTAINS_PATH(o.status, 'one', '$.$cancelled') = 0")

            // The offer must not be in the past
            ->andHaving('MIN(d.begin) > CURRENT_TIMESTAMP()')

            // The offer must take place no earlier than the notice period
            ->andHaving('MIN(d.begin) <= :date')
            ->setParameter('date', $noticePeriod, Types::DATETIME_MUTABLE)

            ->getQuery()
            ->execute()
        ;

        foreach ($attendances as $attendance) {
            $this->commandBus->dispatch(new RemindAttendance($attendance->getId()));
        }
    }
}
