<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Cron;

use Contao\CoreBundle\Cron\Cron;
use Contao\CoreBundle\DependencyInjection\Attribute\AsCronJob;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Message\OfferCompleteNotify;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Workflow\WorkflowInterface;

#[AsCronJob('daily')]
class NotifyOfferComplete
{
    public function __construct(private readonly OfferRepositoryInterface $offers, private readonly MessageBusInterface $eventBus, private readonly WorkflowInterface $offerWorkflow)
    {
    }

    public function __invoke(string $scope): void
    {
        if (Cron::SCOPE_WEB === $scope) {
            return;
        }

        $completed = OfferInterface::STATE_COMPLETED;

        /** @var QueryBuilder $qb */
        $qb = $this->offers->createQueryBuilder('o');

        /** @var Collection<OfferInterface> $offers */
        $offers = $qb
            ->innerJoin('o.dates', 'd')
            ->groupBy('o.id')

            // LEFT JOIN event logs, because we actually want to filter out attendances with event log record.
            ->leftJoin('o.messengerLogs', 'm', Join::WITH, 'm.message = :message')
            ->setParameter('message', OfferCompleteNotify::class)
            ->andWhere('m IS NULL')

            // Must not be completed yet
            ->andWhere("JSON_CONTAINS_PATH(o.status, 'one', '$.$completed') <> 1")

            // The offer must be in the past
            ->andHaving('MAX(d.end) < CURRENT_TIMESTAMP()')

            // Must be completable
            ->innerJoin('o.edition', 'e')
            ->andWhere('e.archived <> 1')
            ->andWhere($qb->expr()->orX('e.surveyWithApplication IS NOT NULL', 'e.surveyWithoutApplication IS NOT NULL'))

            ->getQuery()
            ->execute()
        ;

        foreach ($offers as $offer) {
            if (!$this->offerWorkflow->can($offer, OfferInterface::TRANSITION_COMPLETE)) {
                continue;
            }

            $this->eventBus->dispatch(new OfferCompleteNotify($offer->getId()));
        }
    }
}
