<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Cron;

use Contao\CoreBundle\Cron\Cron;
use Contao\CoreBundle\DependencyInjection\Attribute\AsCronJob;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Message\ApplicationsClosed;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Component\Messenger\MessageBusInterface;

#[AsCronJob('hourly')]
class NotifyApplicationsClosed
{
    public function __construct(private readonly OfferRepositoryInterface $offers, private readonly MessageBusInterface $eventBus)
    {
    }

    public function __invoke(string $scope): void
    {
        if (Cron::SCOPE_WEB === $scope) {
            return;
        }

        $completed = OfferInterface::STATE_COMPLETED;

        /** @var QueryBuilder $qb */
        $qb = $this->offers->createQueryBuilder('o');

        /** @var Collection<OfferInterface> $offers */
        $offers = $qb
            ->innerJoin('o.dates', 'd')
            ->groupBy('o.id')

            // LEFT JOIN event logs, because we actually want to filter out attendances with event log record.
            ->leftJoin('o.messengerLogs', 'm', Join::WITH, 'm.message = :message')
            ->setParameter('message', ApplicationsClosed::class)
            ->andWhere('m IS NULL')

            // Must not be completed yet
            ->andWhere("JSON_CONTAINS_PATH(o.status, 'one', '$.$completed') <> 1")

            // Application deadline must be past
            ->andWhere($qb->expr()->andX(
                'o.applicationDeadline IS NOT NULL',
                'o.applicationDeadline <= CURRENT_TIMESTAMP()'
            ))

            // The offer must be in the future
            ->andHaving('MIN(d.begin) > CURRENT_TIMESTAMP()')

            ->getQuery()
            ->execute()
        ;

        foreach ($offers as $offer) {
            $this->eventBus->dispatch(new ApplicationsClosed($offer->getId()));
        }
    }
}
