<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\Controller;

use Contao\CoreBundle\Filesystem\VirtualFilesystemException;
use Contao\CoreBundle\Filesystem\VirtualFilesystemInterface;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\ContainerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Component\HttpFoundation\UriSigner;
use Symfony\Component\HttpKernel\Attribute\MapQueryParameter;
use Symfony\Component\Routing\Attribute\Route;

#[Route(path: '/storage')]
final class StorageFileController extends AbstractController
{
    public function __construct(
        private readonly ContainerInterface $locator,
        private readonly UriSigner $signer,
    ) {
    }

    #[Route(path: '/{storage}/{path}', name: 'storage_public_path', requirements: ['path' => '.+'])]
    public function __invoke(string $storage, string $path, #[MapQueryParameter('t')] string $validThrough, Request $request): Response
    {
        if (!$this->signer->checkRequest($request)) {
            throw $this->createAccessDeniedException('Hash invalid');
        }

        if (new \DateTimeImmutable($validThrough) < new \DateTimeImmutable()) {
            throw $this->createNotFoundException('Time invalid');
        }

        try {
            $storage = $this->getStorage($storage);
        } catch (ContainerExceptionInterface) {
            throw $this->createNotFoundException('Storage not found');
        }

        try {
            $file = $storage->get($path);
        } catch (VirtualFilesystemException) {
            throw $this->createNotFoundException('File not found');
        }

        $stream = $storage->readStream($path);

        $response = new StreamedResponse(
            function () use ($stream) {
                $outputStream = fopen('php://output', 'w');

                stream_copy_to_stream($stream, $outputStream);
            }
        );

        $response->headers->set('Content-Transfer-Encoding', 'binary');
        $response->headers->set('Content-Type', $file->getMimeType());
        $response->headers->set('Content-Length', (string) fstat($stream)['size']);

        return $response;
    }

    /**
     * @throws ContainerExceptionInterface
     */
    private function getStorage(string $storage): VirtualFilesystemInterface
    {
        return $this->locator->get($storage);
    }
}
