<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\ConsentManager;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\Consent;
use Ferienpass\CoreBundle\Entity\ConsentForm;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\ConsentFormRepository;
use Ferienpass\CoreBundle\Repository\ConsentRepository;

class ConsentManager
{
    public function __construct(private ConsentFormRepository $consentFormRepository, private ConsentRepository $consentRepository, private EntityManagerInterface $entityManager)
    {
    }

    public function sign(ConsentForm $consentForm, User $user, ?ParticipantInterface $participant = null): void
    {
        /** @var Consent|null $signature */
        $signature = $this->consentRepository->findOneBy([
            'consentForm' => $consentForm,
            'user' => $user,
            'participant' => $participant,
        ]);

        if ($signature && !$signature->isValid($consentForm->getText())) {
            $signature = null;
        }

        if ($signature) {
            $signature->setDecision(1);
        } else {
            $signature = new Consent($user, participant: $participant, consentForm: $consentForm);
            $this->entityManager->persist($signature);
        }

        $this->entityManager->flush();
    }

    public function decline(ConsentForm $consentForm, User $user, ?ParticipantInterface $participant = null): void
    {
        /** @var Consent|null $signature */
        $signature = $this->consentRepository->findOneBy([
            'consentForm' => $consentForm,
            'user' => $user,
            'participant' => $participant,
        ]);

        if ($signature && !$signature->isValid($consentForm->getText())) {
            $signature = null;
        }

        if ($signature) {
            $signature->setDecision(0);
        } else {
            $signature = new Consent($user, participant: $participant, consentForm: $consentForm, decision: 0);
            $this->entityManager->persist($signature);
        }

        $this->entityManager->flush();
    }

    /**
     * @param ParticipantInterface[] $participants
     *
     * @return ConsentState[]
     */
    public function forOfferAndParticipants(OfferInterface $offer, array $participants, ?User $user = null): array
    {
        $user ??= (!empty($participants) ? $participants[0]->getUser() : null);

        $consentForms = $this->consentFormRepository->findByOfferOrEdition($offer, $offer->getEdition(), $user, $participants);

        $states = [];
        foreach ($consentForms as $consentForm) {
            $isAccount = 'account' === $consentForm->getSubject();

            foreach ($isAccount ? [null] : $participants as $participant) {
                $signature = $consentForm->getSignatures()->findFirst(fn (mixed $key, Consent $s) => $s->getParticipant() === $participant);

                $states[] = new ConsentState($consentForm, $signature, $participant);
            }
        }

        return $states;
    }
}
