<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\ApplicationSystem\Lotting;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;

class Seat
{
    /**
     * @param Attendance[] $attendances
     */
    public function __construct(private readonly array $attendances, private readonly array $postalCodes = [])
    {
    }

    public function getAttendances(): array
    {
        return $this->attendances;
    }

    public function getSeats(): int
    {
        return \count($this->attendances);
    }

    public function setStatus(string $status): void
    {
        foreach ($this->attendances as $attendance) {
            $attendance->setStatus($status);
        }
    }

    public function getValue(): float
    {
        return array_sum(
            array_map(function (Attendance $attendance): float {
                if ('firstcome' === $attendance->getTask()?->getType()) {
                    return 0.0;
                }

                // Got a seat in a date variant, does not require the current seat
                if ($attendance->getParticipant()->getAttendancesNotHidden(includedCancelled: false)->filter(fn (Attendance $a) => $a->isConfirmed() && $a->getOffer()->isVariantOf($attendance->getOffer()))->count()) {
                    return 0.0;
                }

                // Participant has unprivileged postal code
                if ($this->postalCodes && $attendance->getParticipant()->getPostalCode() && !\in_array($attendance->getParticipant()->getPostalCode(), $this->postalCodes, true)) {
                    return 0.0;
                }

                $offerPopularity = fn (OfferInterface $o): float => log($o->getAttendancesNotHidden()->count());

                $attendancesInTheDraw = $attendance->getParticipant()->getAttendancesNotHidden($attendance->getOffer()->getEdition(), false)->filter(fn (Attendance $a) => $a->getOffer()->getMaxParticipants() && $a->getOffer()->getAttendancesNotHidden()->count() > $a->getOffer()->getMaxParticipants() && (!$a->getTask()?->getMaxApplications() || ($a->getUserPriority() && $a->getUserPriority() < $a->getTask()->getMaxApplications())));
                $waitlistedAttendancesInTheDrawWeighted = $attendancesInTheDraw->map(fn (Attendance $a): float => $a->isWaitlisted() ? $offerPopularity($a->getOffer()) : 0.0);
                if (!$attendancesInTheDraw->count()) {
                    return 0.0;
                }

                $xBadLuckOfTheDraw = array_sum($waitlistedAttendancesInTheDrawWeighted->toArray()) / $attendancesInTheDraw->count();

                $xOnlyFewApplications = min(3 - $attendancesInTheDraw->count(), 0) / 2;

                // TODO correct priority for rejected offers
                $max = $attendance->getTask()?->getMaxApplications() ?: 10;
                $userPriority = min($attendance->getUserPriority() ?: $max + 1, $max + 1);
                $userPriority = ($max + 2) - $userPriority;
                $xUserPriority = $userPriority / ($max + 1);

                return (0.85 * $xUserPriority) + (0.5 * $xBadLuckOfTheDraw) + (0.6 * $xOnlyFewApplications);
            }, $this->attendances)
        );
    }
}
