<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\ApplicationSystem\Lotting;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\FriendCode;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

class Assignment
{
    public function __construct(#[Autowire(param: 'ferienpass.preferred_postal_codes')] private readonly array $postalCodes)
    {
    }

    public function drawForOffer(OfferInterface $offer, ?int $maxWaitlist = null): void
    {
        $seats = new Seats();
        $attendances = $offer->getAttendancesWaiting();
        foreach ($attendances as $attendance) {
            $participant = $attendance->getParticipant();
            if ($seats->contains($participant)) {
                continue;
            }

            /** @var FriendCode $friendCode */
            $friendCode = $participant->getFriendCodes()->findFirst(fn (int $i, FriendCode $c) => $c->getOffer() === $offer);
            $friends = $friendCode?->getParticipants();

            if ($friends) {
                $seats[] = new Seat($attendances->filter(fn (Attendance $a) => $friends->contains($a->getParticipant()))->toArray(), $this->postalCodes ?? []);
            } else {
                $seats[] = new Seat([$attendance], $this->postalCodes ?? []);
            }
        }

        [, $seatIndices] = $this->solveKnapsack($seats, $offer->getMaxParticipants());

        foreach ($seats as $i => $seat) {
            if (\in_array($i, $seatIndices, true)) {
                $seat->setStatus(Attendance::STATUS_CONFIRMED);

                continue;
            }

            if (null === $maxWaitlist) {
                $seat->setStatus(Attendance::STATUS_WAITLISTED);
            } elseif ($maxWaitlist) {
                $seat->setStatus(Attendance::STATUS_WAITLISTED);

                $maxWaitlist -= $seat->getSeats();
            } else {
                $seat->setStatus(Attendance::STATUS_UNFULFILLED);
            }
        }
    }

    /**
     * @param Seat[] $seats
     *
     * @return array<float, array<int>>
     */
    private function solveKnapsack(Seats $seats, $availableSeats, ?int $i = null, ?array &$memo = null): array
    {
        $i ??= \count($seats) - 1;
        $memo ??= [];

        if (isset($memo[$i][$availableSeats])) {
            return [$memo[$i][$availableSeats], $memo['picked'][$i][$availableSeats]];
        }

        // At end of decision branch
        if (0 === $i) {
            if ($seats[$i]->getSeats() <= $availableSeats) {
                $memo[$i][$availableSeats] = $seats[$i]->getValue();
                $memo['picked'][$i][$availableSeats] = [$i];

                return [$seats[$i]->getValue(), [$i]];
            }

            $memo[$i][$availableSeats] = 0;
            $memo['picked'][$i][$availableSeats] = [];

            return [0, []];
        }

        // Get the result of the next branch (without this one)
        [$resultingValueWithoutCurrent, $pickedIndicesWithoutCurrent] = $this->solveKnapsack($seats, $availableSeats, $i - 1, $memo);

        // Skip current seat if to big
        if ($seats[$i]->getSeats() > $availableSeats) {
            $memo[$i][$availableSeats] = $resultingValueWithoutCurrent;
            $memo['picked'][$i][$availableSeats] = $pickedIndicesWithoutCurrent;

            return [$resultingValueWithoutCurrent, $pickedIndicesWithoutCurrent];
        }

        // Get the result of the next branch (WITH this one picked, so available weight is reduced)
        [$resultingValueWithCurrent, $pickedIndicesWithCurrent] = $this->solveKnapsack($seats, $availableSeats - $seats[$i]->getSeats(), $i - 1, $memo);
        $resultingValueWithCurrent += $seats[$i]->getValue();

        // Skip or include current seat
        if ($resultingValueWithCurrent > $resultingValueWithoutCurrent) {
            $resultingValue = $resultingValueWithCurrent;
            $pickedIndices = $pickedIndicesWithCurrent;
            $pickedIndices[] = $i;
        } else {
            $resultingValue = $resultingValueWithoutCurrent;
            $pickedIndices = $pickedIndicesWithoutCurrent;
        }

        $memo[$i][$availableSeats] = $resultingValue;
        $memo['picked'][$i][$availableSeats] = $pickedIndices;

        return [$resultingValue, $pickedIndices];
    }
}
