<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CoreBundle\ApiPlatform;

use ApiPlatform\Doctrine\Orm\Extension\QueryCollectionExtensionInterface;
use ApiPlatform\Doctrine\Orm\Util\QueryNameGeneratorInterface;
use ApiPlatform\Metadata\Operation;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

class OfferAnonymousReadExtension implements QueryCollectionExtensionInterface
{
    public function __construct(#[Autowire(service: 'database_connection')] private readonly Connection $db)
    {
    }

    public function applyToCollection(QueryBuilder $queryBuilder, QueryNameGeneratorInterface $queryNameGenerator, string $resourceClass, ?Operation $operation = null, array $context = []): void
    {
        if (!is_a($resourceClass, OfferInterface::class, true)) {
            return;
        }

        // Is not an anonymous user
        if (\in_array('api:offer:read', $context['groups'] ?? [], true)) {
            return;
        }

        $rootAlias = $queryBuilder->getRootAliases()[0];

        // Fetch editions that are allowed to show
        $editions = $this->db
            ->executeQuery("SELECT edition FROM tl_page WHERE type='offer_list' AND published=1 AND (start='' OR start < UNIX_TIMESTAMP()) AND (stop='' OR stop > UNIX_TIMESTAMP())")
            ->fetchFirstColumn()
        ;

        $queryBuilder
            ->andWhere(\sprintf('%s.edition IN (:editions)', $rootAlias))
            ->setParameter('editions', $editions, ArrayParameterType::INTEGER)
        ;

        // Only show published items
        $published = OfferInterface::STATE_PUBLISHED;
        $queryBuilder
            ->andWhere(\sprintf("JSON_CONTAINS_PATH(%s.status, 'one', '$.%s') = 1", $rootAlias, $published))
        ;
    }
}
