<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Controller\Page;

use Contao\ArticleModel;
use Contao\CoreBundle\DependencyInjection\Attribute\AsPage;
use Contao\CoreBundle\Exception\InsufficientAuthenticationException;
use Contao\CoreBundle\Exception\PageNotFoundException;
use Contao\CoreBundle\Filesystem\VirtualFilesystemInterface;
use Contao\CoreBundle\Routing\ResponseContext\ResponseContextAccessor;
use Contao\PageModel;
use Ferienpass\CmsBundle\Controller\AbstractController;
use Ferienpass\CmsBundle\Fragment\FragmentReference;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\OfferAttachment;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\StreamedResponse;

#[AsPage('offer_list')]
class OfferListPage extends AbstractController
{
    public function __construct(private readonly OfferRepositoryInterface $offers, private readonly VirtualFilesystemInterface $attachmentsStorage)
    {
    }

    public function __invoke(?string $auto_item, Request $request, ResponseContextAccessor $responseContextAccessor): Response
    {
        $this->initializeContaoFramework();

        //        $responseContext = $responseContextAccessor->getResponseContext();

        $pageModel = $request->attributes->get('pageModel');

        //        $edition = $offer->getEdition();
        //        if (null !== $edition && $edition->getActiveTasks('show_offers')->isEmpty()) {
        //            throw new PageNotFoundException();
        //        }

        if (null !== $auto_item) {
            return $this->handleOfferDetails($auto_item, $request, $pageModel);
        }

        return $this->handleOfferList($pageModel);
    }

    private function handleOfferDetails(string $alias, Request $request, PageModel $pageModel): Response
    {
        if (!($offer = $this->offers->findByAlias($alias)) instanceof OfferInterface) {
            throw new PageNotFoundException();
        }

        if (!$request->attributes->getBoolean('preview') && !$offer->isPublished()) {
            throw new PageNotFoundException();
        }

        //        $htmlHeadBag = $responseContext->get(HtmlHeadBag::class);
        //        $htmlHeadBag->setTitle($offer->getName());
        //        if ($date = $offer->getDates()->first()) {
        //            $htmlHeadBag->setTitle(sprintf('%s am %s', $htmlHeadBag->getTitle(), $date->getBegin()->format('d.m.Y')));
        //        }

        if ($request->query->has('file')) {
            return $this->handleDownload($request, $offer);
        }

        if ($request->query->has('login')) {
            $user = $this->getUser();
            if (!$user instanceof User) {
                throw new InsufficientAuthenticationException();
            }

            return $this->redirectToRoute('cms', ['to' => 'offer_list', 'auto_item' => $offer->getAlias()]);
        }

        return $this->createPageBuilder($pageModel)
            ->addFragment('main', new FragmentReference('ferienpass.fragment.offer_details', ['offer' => $offer]))
            ->getResponse()
        ;
    }

    private function handleOfferList(PageModel $pageModel): Response
    {
        $pageBuilder = $this->createPageBuilder($pageModel);

        $articles = ArticleModel::findPublishedByPidAndColumn($pageModel->id, 'main');
        while (null !== $articles && $articles->next()) {
            $pageBuilder->addFragment('main', new FragmentReference('ferienpass.fragment.article', ['id' => $articles->id]));
        }

        $pageBuilder->addFragment('main', new FragmentReference('ferienpass.fragment.offer_list'));

        return $pageBuilder->getResponse();
    }

    private function handleDownload(Request $request, OfferInterface $offer): Response
    {
        $path = $request->query->get('file');

        if (!array_any($offer->getAttachments()->toArray(), fn (OfferAttachment $a) => $path === $a->getAttachment()->getPath())) {
            throw new PageNotFoundException('Invalid download');
        }

        $file = $this->attachmentsStorage->get($path);
        $stream = $this->attachmentsStorage->readStream($path);

        $response = new StreamedResponse(
            function () use ($stream) {
                $outputStream = fopen('php://output', 'w');

                stream_copy_to_stream($stream, $outputStream);
            }
        );

        $response->headers->set('Content-Transfer-Encoding', 'binary');
        $response->headers->set('Content-Type', $file->getMimeType());
        $response->headers->set('Content-Length', (string) fstat($stream)['size']);

        return $response;
    }
}
