<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Components;

use Contao\CoreBundle\OptIn\OptInInterface;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\Common\Collections\Order;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CmsBundle\Application\AbstractParticipantExclusionReason;
use Ferienpass\CmsBundle\Application\ExclusionReasonEvent;
use Ferienpass\CmsBundle\Application\ParticipantExclusionReason;
use Ferienpass\CmsBundle\Dto\AccessCodeDto;
use Ferienpass\CmsBundle\Dto\FriendCodeDto;
use Ferienpass\CoreBundle\ApplicationSystem\ApplicationSystemInterface;
use Ferienpass\CoreBundle\ApplicationSystem\ApplicationSystems;
use Ferienpass\CoreBundle\ConsentManager\ConsentManager;
use Ferienpass\CoreBundle\ConsentManager\ConsentState;
use Ferienpass\CoreBundle\Entity\AccessCodeStrategy;
use Ferienpass\CoreBundle\Entity\AgreementLetterSignature;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\ConsentForm;
use Ferienpass\CoreBundle\Entity\EditionTask;
use Ferienpass\CoreBundle\Entity\FriendCode;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Entity\PostalAddress;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Facade\AttendanceFacade;
use Ferienpass\CoreBundle\Facade\PaymentsFacade;
use Ferienpass\CoreBundle\Message\DeleteAttendance;
use Ferienpass\CoreBundle\Repository\AgreementLetterSignaturesRepository;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\EditionTaskRepository;
use Ferienpass\CoreBundle\Repository\FriendCodeRepository;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Ferienpass\CoreBundle\Validator\ValidAccessCode;
use Ferienpass\CoreBundle\Validator\ValidFriendCode;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Translation\TranslatableMessage;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\ValidatableComponentTrait;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;
use Symfony\UX\TwigComponent\Attribute\PostMount;

#[AsLiveComponent(template: '@Contao/components/application_form.html.twig', route: 'live_component_cms')]
class ApplicationForm extends AbstractController
{
    use DefaultActionTrait;
    use ValidatableComponentTrait;

    #[LiveProp]
    public OfferInterface $offer;

    #[LiveProp]
    public ?ParticipantInterface $participant = null;

    #[LiveProp(writable: true)]
    public bool $doubleBooking = false;

    #[LiveProp]
    public bool $hasNewParticipant = false;

    #[Assert\Valid]
    #[LiveProp(writable: ['firstname', 'lastname', 'email', 'mobile', 'dateOfBirth'])]
    public ?ParticipantInterface $newParticipant = null;

    #[ValidAccessCode]
    #[LiveProp(writable: ['code'])]
    public ?AccessCodeDto $accessCode = null;

    #[ValidFriendCode]
    #[LiveProp(writable: ['code'], onUpdated: ['code' => 'onFriendCodeUpdated'])]
    public ?FriendCodeDto $friendCode = null;

    #[LiveProp(writable: true)]
    public bool $useFriendsBooking = false;

    #[LiveProp(writable: true)]
    public string $useFriendCode = 'on-hold';

    #[LiveProp(writable: true)]
    public array $extra = [];

    #[LiveProp(writable: true)]
    public array $decisions = [];

    #[LiveProp]
    public string $routeParameters;

    #[LiveProp(writable: true, url: true)]
    public string $token = '';

    public ?TranslatableMessage $confirm = null;

    public function __construct(private readonly ApplicationSystems $applicationSystems, private readonly AttendanceRepository $attendances, private readonly ParticipantRepositoryInterface $participantRepository, private readonly FriendCodeRepository $friendCodes, private readonly RequestStack $requestStack, private readonly AgreementLetterSignaturesRepository $signatures, private readonly OptInInterface $optIn, private readonly EventDispatcherInterface $dispatcher, #[Autowire(param: 'ferienpass.friends_booking')] public readonly bool $enableFriendsBooking, #[Autowire(param: 'kernel.secret')] private readonly string $secret, #[Autowire(param: 'ferienpass.withdraw_grace_period')] private readonly ?string $gracePeriod, #[Autowire(param: 'ferienpass.require_postal_address')] private readonly bool $requiresPostalAddress, private readonly PaymentsFacade $paymentsFacade, private readonly EditionTaskRepository $periods, private readonly ConsentManager $consentManager)
    {
    }

    public function mount(OfferInterface $offer, ?AccessCodeStrategy $accessCodeStrategy = null): void
    {
        if ($accessCodeStrategy instanceof AccessCodeStrategy) {
            $this->accessCode = new AccessCodeDto($accessCodeStrategy, $this->accessCode?->getCode());
        }

        $this->offer = $offer;
        $this->friendCode = new FriendCodeDto($offer, $this->friendCode?->getCode());

        foreach ($offer->getApplicationExtra() ?? [] as $extra) {
            $this->extra[$extra['uuid']] ??= '';
        }
    }

    #[ExposeInTemplate]
    public function attendancesDue(): array
    {
        $user = $this->getUser();

        // Guest visitor
        if (!$user instanceof User) {
            return [];
        }

        return $this->paymentsFacade->attendancesDue($user);
    }

    #[ExposeInTemplate]
    public function paymentPeriod(): ?EditionTask
    {
        return $this->periods->payment();
    }

    #[PostMount]
    public function postMount(): void
    {
        if ($this->token && ($optInToken = $this->optIn->find($this->token)) && $optInToken->isValid()) {
            $optInToken->confirm();
            $this->token = '';
        }
    }

    public function onFriendCodeUpdated(): void
    {
        if ($this->friendCode?->getCode()) {
            $this->useFriendCode = 'redeem';
        }
    }

    #[ExposeInTemplate]
    public function participants(): array
    {
        /** @var QueryBuilder $qb */
        $qb = $this->participantRepository->createQueryBuilder('p');

        $user = $this->getUser();
        if ($user instanceof \Symfony\Component\Security\Core\User\UserInterface) {
            $qb
                ->andWhere('p.user = :user')
                ->setParameter('user', $user)
            ;
        } else {
            $sessionParticipants = $this->requestStack->getSession()->isStarted() ? $this->requestStack->getSession()->get('participant_ids') : [];

            $qb
                ->andWhere('p.id IN (:ids)')
                ->setParameter('ids', $sessionParticipants)
            ;
        }

        return $qb->getQuery()->getResult();
    }

    #[ExposeInTemplate]
    public function mustAddPostalAddress(): bool
    {
        $user = $this->getUser();
        if (!$user instanceof User) {
            return false;
        }

        /** @var PostalAddress|false $postalAddress */
        $postalAddress = $user->getPostalAddresses()->first();

        return $this->requiresPostalAddress && (false === $postalAddress || !$postalAddress->getStreet() || !$postalAddress->getPostalCode() || !$postalAddress->getCity());
    }

    #[ExposeInTemplate]
    public function newPriority(): ?int
    {
        if (!$this->participant instanceof ParticipantInterface) {
            return null;
        }

        /** @var Attendance|false $lastAttendanceParticipant */
        $lastAttendanceParticipant = $this->participant
            ?->getAttendancesWaiting()
            ?->matching(Criteria::create()->orderBy(['user_priority' => Order::Ascending]))
            ?->last()
        ;

        $task = $this->applicationSystem()->getTask();
        $priority = $lastAttendanceParticipant ? $lastAttendanceParticipant->getUserPriority() + 1 : 1;
        if ($maxApplications = $task?->getMaxApplications()) {
            $priority = min($maxApplications + 1, $priority);
        }

        return $priority;
    }

    #[ExposeInTemplate]
    public function inGracePeriod(): bool
    {
        $gracePeriod = $this->gracePeriod ? new \DateTimeImmutable($this->gracePeriod) : null;

        return !$gracePeriod instanceof \DateTimeImmutable || ($this->offer->getDates()->isEmpty() || $this->offer->getDates()->first()->getBegin() > $gracePeriod);
    }

    #[ExposeInTemplate]
    public function isRegistered(): bool
    {
        $user = $this->getUser();

        return $user instanceof User && $this->offer->getAttendancesNotHidden()
                ->filter(fn (Attendance $a) => $a->getParticipant()?->getUser() === $user)
                ->count() > 0;
    }

    public function attendance(?ParticipantInterface $participant): ?Attendance
    {
        if (!$participant instanceof ParticipantInterface) {
            return null;
        }

        return $this->offer
            ->getAttendancesNotHidden()
            ->findFirst(fn (int $i, Attendance $a) => $a->getParticipant()->getId() === $participant->getId())
        ;
    }

    public function exclusionReason(ParticipantInterface $participant): ?ParticipantExclusionReason
    {
        $event = new ExclusionReasonEvent($this->offer, $participant, $this->applicationSystem());

        try {
            $this->dispatcher->dispatch($event);
        } catch (AbstractParticipantExclusionReason $reason) {
            return $reason;
        }

        return null;
    }

    #[ExposeInTemplate]
    public function applicationSystem(): ?ApplicationSystemInterface
    {
        return $this->applicationSystems->findApplicationSystem($this->offer);
    }

    #[ExposeInTemplate]
    public function vacant(): ?int
    {
        $countParticipants = $this->attendances->count(['status' => 'confirmed', 'offer' => $this->offer]) + $this->attendances->count(['status' => 'waitlisted', 'offer' => $this->offer]);
        $vacant = $this->offer->getMaxParticipants() > 0 ? $this->offer->getMaxParticipants() - $countParticipants : null;

        return null === $vacant ? null : max(0, $vacant);
    }

    #[ExposeInTemplate]
    public function yourFriendCode(): string
    {
        if (!$this->participant instanceof ParticipantInterface) {
            return '';
        }

        $token = strtoupper(sha1(\sprintf('%d-%d-%s', $this->offer->getId(), $this->participant->getId(), $this->secret)));
        $token = substr($token, 0, 4);

        return \sprintf('%04d-%s', $this->offer->getId(), $token);
    }

    /**
     * @return array<ParticipantInterface>
     */
    public function friendCodeParticipants(): array
    {
        /** @var FriendCode|null $code */
        $code = $this->friendCodes->findOneBy(['offer' => $this->friendCode->getOffer(), 'code' => $this->friendCode->getCode()]);

        return $code?->getParticipants()->toArray() ?? [];
    }

    /**
     * @return FriendCode[]
     */
    public function siblingCodes(): array
    {
        return $this->friendCodes->createQueryBuilder('friendCode')
            ->where('friendCode.offer = :offer')
            ->innerJoin('friendCode.participants', 'participant')
            ->innerJoin('participant.user', 'user')
            ->andWhere('user.id = :userId')
            ->setParameter('offer', $this->offer)
            ->setParameter('userId', $this->getUser())
            ->getQuery()
            ->getResult()
        ;
    }

    /**
     * @return Collection<ParticipantInterface>
     */
    public function siblingsWithoutCode(): Collection
    {
        return $this->offer->getAttendancesNotHidden()
            ->filter(fn (Attendance $a) => $a->getParticipant()->getUser() === $this->getUser())
            ->filter(fn (Attendance $a) => $a->getParticipant()->getId() !== $this->participant?->getId())
            ->filter(fn (Attendance $a) => $a->getParticipant()->getFriendCodes()->filter(fn (FriendCode $fc) => $fc->getOffer() === $this->offer)->isEmpty())
            ->map(fn (Attendance $a) => $a->getParticipant())
        ;
    }

    /**
     * @return Collection<Attendance>
     */
    public function variantApplications(): Collection
    {
        $offerIds = $this->offer->getVariants(include: true)->map(fn (OfferInterface $o) => $o->getId())->toArray();

        return $this->participant->getAttendancesNotHidden(includedCancelled: false)
            ->filter(fn (Attendance $a) => $a->getOffer() !== $this->offer && \in_array($a->getOffer()->getId(), $offerIds, true))
        ;
    }

    #[ExposeInTemplate]
    public function mustSignFirst(): bool
    {
        return null !== $this->consentForm();
    }

    #[ExposeInTemplate]
    public function isAgreementLetterSigned(): bool
    {
        $user = $this->getUser();
        if (!$user instanceof User) {
            return false;
        }

        return $this->signatures->findValidForEdition($this->offer->getEdition(), $user) instanceof AgreementLetterSignature;
    }

    /**
     * @return ConsentState[]
     */
    #[ExposeInTemplate]
    public function requiredConsents(): array
    {
        return $this->consentManager->forOfferAndParticipants($this->offer, $this->participants(), $this->getUser() instanceof User ? $this->getUser() : null);
    }

    #[LiveListener('consentToggled')]
    public function toggleConsent(#[LiveArg] $participant)
    {
        $initialState = 'opt-out' === $this->consentForm()?->getCompulsory();
        $this->decisions[$participant] = !($this->decisions[$participant] ?? $initialState);
    }

    #[ExposeInTemplate]
    public function consentStates(): ?array
    {
        // When there a multiple consent forms to sign, get the first one
        return $this->consentStatesGroupedByForm()[0] ?? null;
    }

    #[ExposeInTemplate]
    public function consentForm(): ?ConsentForm
    {
        /** @var ConsentState|null $consentState */
        $consentState = $this->consentStates()[0] ?? null;

        return $consentState?->getConsentForm();
    }

    #[LiveAction]
    public function sign(#[LiveArg] string $consentId, #[CurrentUser] User $user, EntityManagerInterface $entityManager): ?RedirectResponse
    {
        $consentForm = $entityManager->getRepository(ConsentForm::class)->find(Uuid::fromString($consentId));
        if (!$consentForm instanceof ConsentForm) {
            return null;
        }

        foreach ($this->decisions as $participantId => $decision) {
            $participantId = 'account' === $participantId ? null : (int) $participantId;
            $participantEntity = array_find($this->participants(), fn (ParticipantInterface $p) => $p->getId() === $participantId);
            if ($participantId && !$participantEntity) {
                continue;
            }

            if ($decision) {
                $this->consentManager->sign($consentForm, $user, $participantEntity);
            } else {
                $this->consentManager->decline($consentForm, $user, $participantEntity);
            }
        }

        $this->decisions = [];

        $entityManager->flush();

        if ($this->offer->getEdition()->hasAgreementLetter()
            && !$this->signatures->findValidForEdition($this->offer->getEdition(), $user) instanceof AgreementLetterSignature) {
            $signature = AgreementLetterSignature::fromEdition($this->offer->getEdition(), $user);
            $entityManager->persist($signature);
            $entityManager->flush();
        }

        return $this->redirectToRoute('cms', ['to' => 'offer_list', 'auto_item' => $this->offer->getAlias()]);
    }

    #[LiveAction]
    public function proceed(#[LiveArg] int $participant): void
    {
        if ($participant === $this->participant?->getId()) {
            $this->participant = null;

            return;
        }

        $this->participant = $this->participantRepository->find($participant);
    }

    #[LiveAction]
    public function new(): void
    {
        $this->newParticipant ??= $this->participantRepository->createNew();
        $this->hasNewParticipant = !$this->hasNewParticipant;
    }

    #[LiveAction]
    public function apply(AttendanceFacade $attendanceFacade, EntityManagerInterface $entityManager): void
    {
        if ($this->mustSignFirst()) {
            return;
        }

        if (!$this->participant instanceof ParticipantInterface) {
            return;
        }

        if ($this->useFriendsBooking) {
            $code = $this->friendCodes->findOneBy(['offer' => $this->offer, 'code' => $this->yourFriendCode()]);
            $code ??= new FriendCode($this->offer, $this->yourFriendCode());

            if ('redeem' === $this->useFriendCode) {
                $this->validateField('friendCode');

                $code = $this->friendCodes->findOneBy(['offer' => $this->offer, 'code' => $this->friendCode?->getCode()]);
                $code ??= new FriendCode($this->offer, $this->friendCode?->getCode());
            } elseif (is_numeric($this->useFriendCode)) {
                /** @var ParticipantInterface|null $sibling */
                $sibling = $this->siblingsWithoutCode()->findFirst(fn (int $i, ParticipantInterface $p) => $p->getId() === (int) $this->useFriendCode);
                $sibling?->addFriendCode($code);
            } elseif ('on-hold' !== $this->useFriendCode) {
                $code = $this->friendCodes->findOneBy(['offer' => $this->offer, 'code' => $this->useFriendCode]);
            }

            $this->participant->addFriendCode($code);
        }

        $attendance = $attendanceFacade->create($this->offer, $this->participant, allAlternateDates: $this->variantApplications()->isEmpty() || $this->doubleBooking);

        if (\count($this->extra)) {
            $attendance->setExtra($this->extra);
            $entityManager->flush();
        }

        $this->confirm = new TranslatableMessage('Die Anmeldungen wurden angenommen.');

        $this->participant = null;
        $this->reset();
    }

    #[LiveAction]
    public function withdraw(MessageBusInterface $commandBus, WorkflowInterface $attendanceStateMachine): void
    {
        $attendance = $this->attendance($this->participant);
        if (!$attendance instanceof Attendance) {
            return;
        }

        if (!$attendanceStateMachine->can($attendance, Attendance::TRANSITION_WITHDRAW)) {
            // Todo, warning; or maybe disallow action in the first place
            return;
        }

        if ($this->applicationSystem() instanceof LotApplicationSystem) {
            $commandBus->dispatch(new DeleteAttendance($attendance->getId()));
        } else {
            $attendanceStateMachine->apply($attendance, Attendance::TRANSITION_WITHDRAW, ['notify' => false]);
        }

        $this->confirm = new TranslatableMessage('Die Anmeldung wurde erfolgreich zurückgezogen');

        $this->participant = null;
        $this->reset();
    }

    #[LiveAction]
    public function saveParticipant(Request $request, EntityManagerInterface $entityManager): void
    {
        $this->validateField('newParticipant');

        $entityManager->persist($participant = $this->newParticipant);

        if ($this->accessCode instanceof AccessCodeDto) {
            $participant->addAccessCode($this->accessCode->toEntity());
        }

        if ($this->getUser() instanceof User) {
            $participant->setUser($this->getUser());
        }

        $entityManager->flush();

        if (!$participant->getUser() instanceof User) {
            $request->getSession()->set('participant_ids', array_unique(array_merge($request->getSession()->get('participant_ids', []), [$participant->getId()])));

            // Verify email address
            $optInToken = $this->optIn->create('apply', $participant->getEmail(), ['Participant' => [$participant->getId()]]);
            $optInToken->send(
                'Bitte bestätigen Sie Ihre E-Mail-Adresse',
                \sprintf(
                    "Bitte bestätigen Sie Ihre E-Mail-Adresse für die Anmeldung beim Ferienpass\n\n\n%s",
                    $this->generateUrl('cms', ['to' => 'offer_list', 'parameters' => $this->routeParameters, 'token' => $optInToken->getIdentifier()], UrlGeneratorInterface::ABSOLUTE_URL)
                )
            );
            $this->confirm = new TranslatableMessage('Teilnehmer:in erfolgreich gespeichert. Bitte bestätigen Sie nur noch die E-Mail, die wir Ihnen gerade zugesandt haben.');
        } else {
            $this->confirm = new TranslatableMessage('Teilnehmer:in erfolgreich gespeichert. Wollen Sie die Person gleich zum Angebot anmelden?');
        }

        $this->reset();
    }

    public function reset(): void
    {
        $this->resetValidation();

        $this->hasNewParticipant = false;
        $this->newParticipant = null;

        $this->useFriendsBooking = false;
        $this->friendCode = new FriendCodeDto($this->offer);
        $this->useFriendCode = 'on-hold';
    }

    private function consentStatesGroupedByForm(): array
    {
        $grouped = [];
        foreach ($this->requiredConsents() as $state) {
            if (!$state->isAcknowledged() || ($state->isCompulsory() && !$state->isSigned())) {
                $grouped[(string) $state->getConsentForm()->getId()][] = $state;
            }
        }

        return array_values($grouped);
    }
}
