<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Application\ExclusionReasons;

use Ferienpass\CmsBundle\Application\ExclusionReasonEvent;
use Ferienpass\CoreBundle\Entity\OfferDate;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\Attribute\AsEventListener;
use Symfony\Component\Translation\TranslatableMessage;

#[AsEventListener(priority: 80)]
class AgeInvalidListener
{
    public function __construct(#[Autowire(param: 'ferienpass.age_tolerance')] private readonly ?string $ageTolerance)
    {
    }

    public function __invoke(ExclusionReasonEvent $event): void
    {
        $offer = $event->getOffer();
        $participant = $event->getParticipant();

        if (!$offer->getMinAge() && !$offer->getMaxAge()) {
            return;
        }

        /** @var OfferDate|false $date */
        $date = $offer->getDates()->first();
        if (false === $date || null === $dateBegin = $date->getBegin()) {
            return;
        }

        $allowedAges = [$participant->getAge($dateBegin)];
        if (null !== $this->ageTolerance) {
            $allowedAges[] = $participant->getAge((clone $dateBegin)->modify(\sprintf('-%s', $this->ageTolerance)));
            $allowedAges[] = $participant->getAge((clone $dateBegin)->modify(\sprintf('+%s', $this->ageTolerance)));
        }

        if ([] === array_filter($allowedAges)) {
            return;
        }

        if (($offer->getMinAge() && max($allowedAges) < $offer->getMinAge()) || ($offer->getMaxAge() && min($allowedAges) > $offer->getMaxAge())) {
            throw new AgeInvalid($offer, $participant, new TranslatableMessage('ineligible.invalidAge', ['name' => $participant->getFirstname()]));
        }
    }
}
