<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Security;

use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Security\Exception\BlockedEmailException;
use Safe\DateTimeImmutable;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\UriSigner;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\DisabledException;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;

class JsonLoginFailureHandler implements AuthenticationFailureHandlerInterface
{
    public function __construct(private readonly UrlGeneratorInterface $urlGenerator, private readonly UriSigner $uriSigner)
    {
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): Response
    {
        $exception = $exception->getPrevious() ?? $exception;
        if ($exception instanceof BlockedEmailException && ($user = $exception->getUser()) && $user instanceof User) {
            return new JsonResponse([
                'error' => 'blocked_email',
                'redirect_to' => $this->uriSigner->sign($this->urlGenerator->generate('cms_blocked_email', [
                    'id' => $user->getId(),
                    'email' => $user->getEmail(),
                    'unsubscribed' => ($user->getBlockedContact()?->getCreatedAt() ?? new DateTimeImmutable())->format('c'),
                    'valid' => new DateTimeImmutable('+5 min')->format('c'),
                ], UrlGeneratorInterface::ABSOLUTE_URL)),
            ], JsonResponse::HTTP_UNAUTHORIZED);
        }

        if ($exception instanceof DisabledException) {
            return new JsonResponse(['error' => 'account_disabled'], JsonResponse::HTTP_UNAUTHORIZED);
        }

        return new JsonResponse(['error' => 'invalid_credentials'], JsonResponse::HTTP_UNAUTHORIZED);
    }
}
