<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Controller\Page;

use Contao\CoreBundle\Exception\PageNotFoundException;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CmsBundle\Controller\AbstractController;
use Ferienpass\CmsBundle\Fragment\FragmentReference;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Message\AccountActivated;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\UriSigner;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Routing\Attribute\Route;

class RegistrationController extends AbstractController
{
    public function __construct(private readonly UriSigner $uriSigner, private readonly Security $security, private readonly MessageBusInterface $eventBus)
    {
    }

    #[Route('/registrierung/ok', name: 'cms_registration_confirm')]
    public function confirm(): Response
    {
        $this->initializeContaoFramework();

        return $this->createPageBuilder()
            ->addFragment('main', new FragmentReference('ferienpass.fragment.registration_confirm'))
            ->getResponse()
        ;
    }

    #[Route('/registrierung/{id}', name: 'cms_registration_activate', requirements: ['id' => '\d+'])]
    public function activate(?User $user, Request $request, EntityManagerInterface $entityManager): Response
    {
        if (!$this->uriSigner->checkRequest($request)) {
            throw new PageNotFoundException();
        }

        if (!$request->query->has('valid') || new \DateTimeImmutable($request->query->get('valid')) < new \DateTimeImmutable()) {
            throw new PageNotFoundException();
        }

        // Cannot find user in database
        if (!$user instanceof User) {
            throw new PageNotFoundException();
        }

        // Already logged in
        $currentUser = $this->getUser();
        if ($currentUser instanceof User && $currentUser->getId() === $user->getId()) {
            return $this->redirectToRoute('cms_registration_welcome');
        }

        $user->setDisabled(false);

        $entityManager->flush();

        $this->eventBus->dispatch(new AccountActivated($user->getId()));

        $this->security->login($user, 'json_login', 'contao_frontend');

        return $this->redirectToRoute('cms_registration_welcome');
    }

    #[Route('/registrierung/willkommen', name: 'cms_registration_welcome')]
    public function welcome(): Response
    {
        $this->initializeContaoFramework();

        $this->checkToken();

        return $this->createPageBuilder()
            ->addFragment('main', new FragmentReference('ferienpass.fragment.registration_welcome'))
            ->getResponse()
        ;
    }

    #[Route('/email-abgemeldet', name: 'cms_blocked_email')]
    public function blockedEmail(Request $request): Response
    {
        $this->initializeContaoFramework();

        if (!$this->uriSigner->check($request->getUri())) {
            throw new PageNotFoundException();
        }

        if (!$request->query->has('valid') || new \DateTimeImmutable($request->query->get('valid')) < new \DateTimeImmutable()) {
            throw new PageNotFoundException();
        }

        return $this->createPageBuilder()
            ->addFragment('main', new FragmentReference('ferienpass.fragment.account_unsubscribed'))
            ->getResponse()
        ;
    }

    #[Route('/account-geloescht', name: 'cms_account_deleted')]
    public function accountDeleted(): Response
    {
        $this->initializeContaoFramework();

        return $this->createPageBuilder()
            ->addFragment('main', new FragmentReference('ferienpass.fragment.account_deleted'))
            ->getResponse()
        ;
    }
}
