<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Controller\Fragment;

use Contao\CoreBundle\Controller\AbstractController;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CmsBundle\Form\CompoundType\PostalAddressType;
use Ferienpass\CmsBundle\Form\SimpleType\ContaoRequestTokenType;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Entity\PaymentItem;
use Ferienpass\CoreBundle\Payments\Provider\PaymentProviderInterface;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpClient\Exception\ClientException;
use Symfony\Component\HttpClient\Exception\ServerException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;

class PayController extends AbstractController
{
    public function __construct(private readonly PaymentProviderInterface $paymentProvider, private readonly EntityManagerInterface $entityManager)
    {
    }

    public function __invoke(Payment $payment, Request $request, Session $session): Response
    {
        if (!$payment->getBillingAddress()) {
            $form = $this->buildForm();

            $form->handleRequest($request);
            if ($form->isSubmitted() && $form->isValid()) {
                $this->updateBillingAddress($form, $payment);

                return $this->redirectToRoute('cms_pay', ['token' => $payment->getUuid()]);
            }

            return $this->render('@Contao/fragment/pay.html.twig', [
                'form' => $form,
                'payment' => $payment,
            ]);
        }

        // A payment should only be considered if the attendances are not paid yet to prevent overpayment
        $isValid = $payment->getItems()->forAll(fn ($k, PaymentItem $i) => !($i->getAttendance()?->isPaid() ?? false) || !$i->getAttendance()->getOffer()->getFee());

        if (!$isValid) {
            return $this->render('@Contao/fragment/pay.html.twig', [
                'payment' => $payment,
            ]);
        }

        try {
            $payUrl = $this->paymentProvider->dispatchPayment($payment);
        } catch (ServerException|ClientException) {
            return $this->render('@Contao/fragment/pay.html.twig', [
                'paymentProviderError' => true,
                'payment' => $payment,
            ]);
        }

        $session->set('fp.processPayment', $payment->getId());

        return $this->render('@Contao/fragment/pay.html.twig', [
            'payment' => $payment,
            'payUrl' => $payUrl,
        ]);
    }

    private function updateBillingAddress(FormInterface $form, Payment $payment): void
    {
        $postalAddress = $form->get('postalAddress')->getData();
        $payment->setBillingAddress(
            <<<EOF
                {$form->get('name')->getData()}
                {$postalAddress->getStreet()}
                {$postalAddress->getPostalCode()} {$postalAddress->getCity()}
                {$postalAddress->getCountry()}
                EOF
        );

        $this->entityManager->flush();
    }

    private function buildForm(): FormInterface
    {
        return $this->createFormBuilder()
            ->add('name', options: ['label' => 'Vor- und Zuname'])
            ->add('postalAddress', PostalAddressType::class, ['label' => 'Rechnungsadresse'])
            ->add('requestToken', ContaoRequestTokenType::class)
            ->add('submit', SubmitType::class, ['label' => 'Speichern'])
            ->getForm()
        ;
    }
}
