<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Controller\Fragment;

use Contao\CoreBundle\Controller\AbstractController;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CmsBundle\Filter\OfferListFilter;
use Ferienpass\CoreBundle\ApplicationSystem\ApplicationSystems;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Pagination\Paginator;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;

final class OfferListController extends AbstractController
{
    public function __construct(private readonly EditionRepository $editions, private readonly OfferRepositoryInterface $offers, private readonly OfferListFilter $filter, private readonly FormFactoryInterface $formFactory, private readonly ApplicationSystems $applicationSystems)
    {
    }

    public function __invoke(?array $ids, Request $request, Session $session): Response
    {
        /** @var QueryBuilder $qb */
        $qb = $this->offers->createQueryBuilder('o');

        self::prepareQueryBuilder($qb);

        if (null !== $ids) {
            $qb->andWhere('o.id IN (:ids)')->setParameter('ids', $ids);
        }

        $hasEditions = null === $ids && $this->editions->count([]) > 0;
        $pageModel = $request->attributes->get('pageModel');
        if ($pageModel?->edition) {
            $edition = $this->editions->find($pageModel->edition);
        } else {
            $edition = $this->editions->findOneToShow($pageModel);
        }

        if ($hasEditions && null !== $edition) {
            $qb->andWhere('o.edition = :edition')->setParameter('edition', $edition);
        }

        if ($hasEditions && (null === $edition || !$edition->isOnline())) {
            return $this->render('@Contao/fragment/offer_list.html.twig');
        }

        if ($edition) {
            $applicationSystem = $this->applicationSystems->findApplicationSystem($edition);
        }

        $filterForm = $this->instantiateForm($request);

        $this->filter->apply($qb, $filterForm);

        $paginator = new Paginator($qb)->paginate($request->query->getInt('page', 1));

        $this->tagResponse(array_map(fn (OfferInterface $offer) => 'offer.'.$offer->getId(), (array) $paginator->getResults()));

        return $this->render('@Contao/fragment/offer_list.html.twig', [
            'filterForm' => $filterForm,
            'edition' => $edition ?? null,
            'pagination' => $paginator,
            'applicationSystem' => $applicationSystem ?? null,
        ]);
    }

    public static function prepareQueryBuilder(QueryBuilder $queryBuilder): QueryBuilder
    {
        $published = OfferInterface::STATE_PUBLISHED;

        return $queryBuilder
            ->leftJoin('o.dates', 'd')
            ->addSelect('d')
            ->where("JSON_CONTAINS_PATH(o.status, 'one', '$.$published') = 1")
            ->groupBy('o.id')
            ->addGroupBy('d.id')
        ;
    }

    protected function instantiateForm(Request $request): FormInterface
    {
        $filterDataFromUrl = array_filter($request->query->all(), fn (string $key) => \in_array($key, $this->filter->getFilterNames(), true), \ARRAY_FILTER_USE_KEY);

        $filterForm = $this->formFactory->create($this->filter::class);
        $filterForm->submit($filterDataFromUrl);

        return $this->formFactory->create($this->filter::class, $filterForm->getData());
    }
}
