<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Components\SignIn;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CmsBundle\Form\UserRegistrationType;
use Ferienpass\CoreBundle\Entity\Participant\BaseParticipant;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Message\AccountAlreadyExists;
use Ferienpass\CoreBundle\Message\AccountCreated;
use Ferienpass\CoreBundle\Message\AccountDisabled;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\ComponentWithFormTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(template: '@Contao/components/sign_in/registration.html.twig', route: 'live_component_cms')]
class Registration extends AbstractController
{
    use ComponentWithFormTrait;
    use DefaultActionTrait;

    private ?User $initialFormData = null;

    public function __construct(private readonly ParticipantRepositoryInterface $participants)
    {
    }

    #[LiveAction]
    public function submit(Session $session, Request $request, EntityManagerInterface $em, MessageBusInterface $eventBus, UserRepository $repository): RedirectResponse
    {
        $this->submitForm();

        /** @var User $user */
        $user = $this->getForm()->getData();

        if (null !== ($existing = $repository->findOneBy(['email' => $user->getEmail()]))) {
            $session->set('registration.email', $user->getEmail());
            if ($existing->isDisabled()) {
                $eventBus->dispatch(new AccountDisabled($existing->getId()));
            } else {
                $eventBus->dispatch(new AccountAlreadyExists($existing->getId()));
            }

            return $this->redirectToRoute('cms_registration_confirm');
        }

        $user->setRoles(['ROLE_MEMBER']);
        $user->setDisabled();

        $session->set('registration.email', $user->getEmail());
        $em->persist($user);

        $this->migrateSessionParticipants($session, $user);

        $em->flush();

        $eventBus->dispatch(new AccountCreated($user->getId(), $request->getHost()));

        return $this->redirectToRoute('cms_registration_confirm');
    }

    protected function instantiateForm(): FormInterface
    {
        // Cast to user, for password hasher
        $this->initialFormData ??= new User();

        return $this->createForm(UserRegistrationType::class, $this->initialFormData);
    }

    private function migrateSessionParticipants(Session $session, User $user): void
    {
        $ids = $session->get('participant_ids', []);
        if (0 === \count($ids)) {
            return;
        }

        foreach ($this->participants->findBy(['id' => $ids]) as $participant) {
            /** @var BaseParticipant $participant */
            if (null === $participant) {
                continue;
            }

            $participant->setUser($user);
        }
    }
}
