<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Components;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\EditionTask;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Entity\Payment as PaymentEntity;
use Ferienpass\CoreBundle\Entity\PaymentItem;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Facade\PaymentsFacade;
use Ferienpass\CoreBundle\Payments\Provider\PaymentProviderInterface;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\PaymentRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\Metadata\UrlMapping;
use Symfony\UX\LiveComponent\ValidatableComponentTrait;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;

#[AsLiveComponent(template: '@Contao/components/payment.html.twig', route: 'live_component_cms')]
class Payment extends AbstractController
{
    use ComponentToolsTrait;
    use DefaultActionTrait;
    use ValidatableComponentTrait;

    #[LiveProp]
    public EditionTask $paymentPeriod;

    #[LiveProp(writable: true, url: new UrlMapping(as: 'step'))]
    public ?string $step = null;

    #[LiveProp(writable: true)]
    public array $selectedItems = [];

    public function __construct(private readonly AttendanceRepository $attendances, private readonly PaymentRepository $payments, private readonly PaymentProviderInterface $paymentProvider, #[Autowire(param: 'ferienpass.payment_allow_select')] private readonly bool $allowSelect, private readonly EventDispatcherInterface $dispatcher, private readonly PaymentsFacade $paymentsFacade)
    {
    }

    #[ExposeInTemplate]
    public function attendancesDue(): array
    {
        $user = $this->getUser();
        if (!$user instanceof User) {
            return [];
        }

        return $this->paymentsFacade->attendancesDue($user);
    }

    #[ExposeInTemplate]
    public function debtorBalance(): int
    {
        $user = $this->getUser();
        if (!$user instanceof User) {
            return 0;
        }

        return $user->getDebtor()->getBalance() ?? 0;
    }

    #[ExposeInTemplate]
    public function requiresPayment(): bool
    {
        return [] === $this->attendancesNonPayable();
    }

    #[ExposeInTemplate]
    public function isSelectable(): bool
    {
        return $this->allowSelect;
    }

    #[ExposeInTemplate]
    public function dispatcher(): EventDispatcherInterface
    {
        return $this->dispatcher;
    }

    public function participants(): array
    {
        $participants = array_map(fn (Attendance $a) => $a->getParticipant(), $this->attendancesDue());

        return array_values(array_intersect_key($participants, array_unique(array_map(fn (ParticipantInterface $p) => $p->getId(), $participants))));
    }

    #[ExposeInTemplate]
    public function attendancesForPayment(): array
    {
        if (!$this->isSelectable()) {
            return $this->attendancesDue();
        }

        $user = $this->getUser();

        return $this->attendances->createQueryBuilder('attendance')
            ->select('attendance', 'participant')
            ->innerJoin('attendance.participant', 'participant')
            ->where('participant.user = :user')
            ->setParameter('user', $user->getId())
            ->andWhere('attendance.id IN (:ids)')
            ->setParameter('ids', $this->selectedItems)
            ->getQuery()
            ->getResult()
        ;
    }

    #[ExposeInTemplate]
    public function totalAmount(): int
    {
        $sum = array_sum(array_map(fn (Attendance $a) => $a->getOffer()->getFeePayable($a->getParticipant(), $this->dispatcher), $this->attendancesForPayment()));

        return $sum - $this->debtorBalance();
    }

    #[LiveListener('proceedTo')]
    public function proceedTo(#[LiveArg] string $step): void
    {
        $this->step = $step;
    }

    #[LiveAction]
    public function redirectToPay(#[CurrentUser] User $user, EntityManagerInterface $entityManager, Session $session): RedirectResponse
    {
        // $this->validate();

        $payableAttendances = $this->attendancesForPayment();

        $user->createDebtor();
        $payment = PaymentEntity::fromAttendances($user->getDebtor(), $payableAttendances, $this->dispatcher, user: $user);

        if (0 !== $user->getDebtor()->getBalance()) {
            $balance = min($user->getDebtor()->getBalance(), $payment->getTotalAmount());
            $payment->addItem(new PaymentItem($payment, (-1) * $balance, type: 'discount'));
        }

        $entityManager->persist($payment);
        $entityManager->flush();

        $redirect = $this->paymentProvider->dispatchPayment($payment);

        $session->set('fp.processPayment', $payment->getId());

        return $this->redirect($redirect);
    }

    #[LiveAction]
    public function confirmNonPayable(EntityManagerInterface $entityManager): void
    {
        /** @var Attendance $item */
        foreach ($this->attendancesNonPayable() as $item) {
            $item->setPaid();
        }

        $entityManager->flush();

        $this->step = 'confirmed';
    }

    private function attendancesNonPayable(): array
    {
        return array_filter($this->attendancesForPayment(), fn (Attendance $attendance) => 0 === $attendance->getOffer()->getFeePayable($attendance->getParticipant(), $this->dispatcher()));
    }
}
