<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\CmsBundle\Components;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\MedicalCareObjection;
use Ferienpass\CoreBundle\Entity\Participant\BaseParticipant;
use Ferienpass\CoreBundle\Entity\PhotoPermission;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;

#[AsLiveComponent(template: '@Contao/components/consents.html.twig', route: 'live_component_cms')]
class Consents extends AbstractController
{
    use DefaultActionTrait;

    public function __construct(private readonly ParticipantRepositoryInterface $participants, private readonly Security $security)
    {
    }

    #[ExposeInTemplate]
    public function participants(): array
    {
        $user = $this->getUser();
        if (!($user instanceof User)) {
            return [];
        }

        return $this->participants->findBy(['user' => $user]);
    }

    #[LiveAction]
    public function consent(#[LiveArg] int $participant, #[CurrentUser] User $user, EntityManagerInterface $entityManager): void
    {
        /** @var BaseParticipant $participant */
        $participant = $this->participants->find($participant);
        $this->denyAccessUnlessGranted('edit', $participant);

        if ($consent = $participant->getConsent('photo_permission')) {
            if ($consent->isRevoked()) {
                $consent->unrevoke();
            } else {
                $consent->revoke();
            }
        } else {
            $entityManager->persist(new PhotoPermission($user, $participant));
        }

        $entityManager->flush();
    }

    #[LiveAction]
    public function objectMed(#[LiveArg] int $participant, #[CurrentUser] User $user, EntityManagerInterface $entityManager): void
    {
        $participant = $this->participants->find($participant);
        $this->denyAccessUnlessGranted('edit', $participant);

        $entityManager->persist(new MedicalCareObjection($user, $participant));

        $entityManager->flush();
    }
}
