<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Menu;

use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Knp\Menu\FactoryInterface;
use Knp\Menu\ItemInterface;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Authentication\Token\SwitchUserToken;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Component\Security\Http\Logout\LogoutUrlGenerator;
use Symfony\Component\Translation\TranslatableMessage;

class MenuBuilder
{
    public function __construct(private readonly FactoryInterface $factory, private readonly LogoutUrlGenerator $logoutUrlGenerator, private readonly AuthorizationCheckerInterface $authorizationChecker, private readonly RequestStack $requestStack, private readonly EditionRepository $editions, private readonly Security $security, private readonly HostRepository $hosts)
    {
    }

    public function primaryNavigation(): ItemInterface
    {
        $menu = $this->factory->createItem('root');

        $edition = $this->editions->findDefaultForHost();
        if ($edition instanceof \Ferienpass\CoreBundle\Entity\Edition && $this->security->isGranted('offer.create', $edition)) {
            $menu->addChild('offers.title', [
                'route' => 'admin_offers_index',
                'routeParameters' => [
                    'edition' => $edition->getAlias(),
                ],
            ]);
        } else {
            $menu->addChild('offers.title', [
                'route' => 'admin_offers_index',
            ]);
        }

        if (!$this->isGranted('ROLE_ADMIN')) {
            $menu->addChild('profile.title', [
                'route' => 'admin_profile_index',
            ]);
        }

        if ($this->isGranted('ROLE_PARTICIPANTS_ADMIN')) {
            $menu->addChild('participants.title', [
                'route' => 'admin_participants_index',
            ]);
        }

        if ($this->isGranted('ROLE_ADMIN')) {
            $menu->addChild('hosts.title', [
                'route' => 'admin_hosts_index',
            ]);
        }
        if ($this->isGranted('ROLE_PAYMENTS_ADMIN')) {
            $menu->addChild('payments.title', [
                'route' => 'admin_payments_index',
            ]);
        }
        if ($this->isGranted('ROLE_ADMIN')) {
            $menu->addChild('accounts.title', [
                'route' => 'admin_accounts_index',
            ]);
            $menu->addChild('files.title', [
                'route' => 'admin_files',
            ]);
            $menu->addChild('tools.title', [
                'route' => 'admin_tools',
            ]);
        }

        return $menu;
    }

    public function userNavigation(): ItemInterface
    {
        $request = $this->requestStack->getMainRequest();
        $menu = $this->factory->createItem('root');

        $menu->addChild('user.title', [
            'route' => 'admin_user_index',
            'current' => $request instanceof \Symfony\Component\HttpFoundation\Request && 'admin_user_index' === $request->attributes->get('_route'),
            'extras' => ['icon' => 'user-circle-filled'],
        ]);

        $menu->addChild('user.password.title', [
            'route' => 'admin_security',
            'current' => $request instanceof \Symfony\Component\HttpFoundation\Request && 'admin_security' === $request->attributes->get('_route'),
            'extras' => ['icon' => 'lock-closed-filled'],
        ]);

        $token = $this->security->getToken();

        if ($token instanceof SwitchUserToken) {
            $impersonatorUser = $token->getOriginalToken()->getUser();
            $menu->addChild('user.impersonation_exit', [
                'label' => new TranslatableMessage('user.impersonation_exit', ['%impersonator%' => $impersonatorUser->getUserIdentifier()], 'admin'),
                'route' => 'admin_accounts_index',
                'routeParameters' => ['_switch_user' => '_exit'],
                'linkAttributes' => ['data-turbo' => 'false'],
                'extras' => ['icon' => 'logout-filled'],
            ]);
        }

        $menu->addChild('user.logout', [
            'uri' => $this->logoutUrlGenerator->getLogoutUrl('ferienpass_admin'),
            'linkAttributes' => ['data-turbo' => 'false'],
            'extras' => ['icon' => 'logout-filled'],
        ]);

        return $menu;
    }

    public function offerFilters(array $options = []): ItemInterface
    {
        $request = $this->requestStack->getMainRequest();
        $menu = $this->factory->createItem('root');

        $user = $this->security->getUser();
        if (!$user instanceof User) {
            return $menu;
        }

        $editionNav = $this->factory->createItem('edition');
        $hostNav = $this->factory->createItem('host');

        $editions = $this->editions->findBy([], ['id' => 'DESC'], 5);
        $defaultEdition = $this->editions->findDefaultForHost();
        foreach ($editions as $edition) {
            $editionNav->addChild((string) $edition->getAlias(), [
                'label' => $edition->getName(),
                'route' => 'admin_offers_index',
                'routeParameters' => ['edition' => $edition->getAlias()] + $request?->query->all() ?? [],
                'current' => $request?->query->has('edition')
                    ? $edition->getAlias() === $request?->query->get('edition')
                    : $edition->getId() === $defaultEdition?->getId(),
            ]);
        }

        $edition = $this->editions->findDefaultForHost();
        foreach ($this->hosts->findByUser($user) as $host) {
            $hostNav->addChild((string) $host->getAlias(), [
                'label' => $host->getName(),
                'route' => 'admin_offers_index',
                'routeParameters' => ['host' => $host->getAlias(), 'edition' => $edition->getAlias()] + $request?->query->all() ?? [],
                'current' => !$request?->query->has('host') || $host->getAlias() === $request?->query->get('host'),
            ]);
        }

        $menu->addChild($editionNav);
        $menu->addChild($hostNav);

        return $menu;
    }

    public function participantListActions(array $options = []): ItemInterface
    {
        $offer = $options['offer'] ?? null;
        if (!$offer instanceof OfferInterface) {
            throw new \InvalidArgumentException('Pass "offer" as option');
        }

        $menu = $this->factory->createItem('root');

        $menu->addChild('pdf', [
            'label' => 'offers.action.participantList',
            'route' => 'admin_offers_participants',
            'routeParameters' => ['uuid' => $offer->getUuid(), '_suffix' => '.pdf'],
            'extras' => [
                'icon' => 'document-download-solid',
            ],
        ]);

        //        if (!$offer->isCancelled()
        //            && ((null === $edition = $offer->getEdition()) || !$edition->getActiveTasks('show_offers')->isEmpty())) {
        //            $menu->addChild('cancel', [
        //                'label' => 'offers.action.cancel',
        //                'route' => 'admin_offer_show',
        //                'routeParameters' => ['id' => $offer->getId(), 'act' => 'cancel'],
        //                'display' => $this->isGranted('cancel', $offer),
        //                'extras' => [
        //                    'method' => 'post',
        //                    'icon' => 'ban-solid',
        //                ],
        //            ]);
        //        }
        //
        //        if ($offer->isCancelled()) {
        //            $menu->addChild('reactivate', [
        //                'label' => 'offers.action.reactivate',
        //                'route' => 'admin_offer_show',
        //                'routeParameters' => ['id' => $offer->getId(), 'act' => 'relaunch', 'edition' => $offer->getEdition()->getAlias()],
        //                'display' => $this->isGranted('reactivate', $offer),
        //                'extras' => [
        //                    'method' => 'post',
        //                    'icon' => 'trash-solid',
        //                ],
        //            ]);
        //        }

        return $menu;
    }

    private function isGranted(string $attribute, mixed $subject = null): bool
    {
        return $this->authorizationChecker->isGranted($attribute, $subject);
    }
}
